#!/bin/sh
# Copyright 2024 Ettus Research, a National Instruments Brand
#
# SPDX-License-Identifier: GPL-3.0-or-later
#

function user_confirmation {
    echo # new line
    read -p "Press \"y\" to continue " -n 1 -r
    echo -e "\n" # two new lines

    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo "aborting"
        exit 1
    fi
}


read -d '' DISCLAIMER <<- EOF
This script updates the microcontroller firmware (RO part). The change is
persistent across power cycles. Incorrect updates may brick the device and may
need you to use the host based update process. This script may take upto a minute to run successfully.

Updating the microcontroller firmware (RO part) is only required if the Ettus
Research support told you to do so.

EOF

echo "$DISCLAIMER"
user_confirmation

ECTOOL=/usr/sbin/ectool
CURRENT_VERSION=$($ECTOOL version)
CURRENT_RO_VERSION=$(echo "$CURRENT_VERSION" | grep "^RO version")
BOARD=$(echo "$CURRENT_RO_VERSION" | sed "s|^RO version:\s*\([a-z0-9\-]*\).*|\1|")

# Fetch the board revision from the motherboard eeprom on the device using eeprom-dump mb command.
EEPROMDUMP=/usr/bin/eeprom-dump
EEPROMDUMP_OUTPUT=$($EEPROMDUMP mb)
COMPAT_REV_HEX=$(echo "$EEPROMDUMP_OUTPUT" | grep "usrp_eeprom_board_info" | sed -n -e 's/^.*compat_rev: 0x\([0-9a-fA-F]*\).*$/\1/p')
COMPAT_REV=$((16#$COMPAT_REV_HEX))


case $BOARD in
    "titanium"|"titanium-rev5")
        let RO_SIZE=0x40000
     ;;
     *)
        echo "unsupported board: $BOARD"
        exit 1
esac

if [ "$BOARD" = "titanium" ]; then
    BASENAME="ec-$BOARD-rev$COMPAT_REV"
else
    BASENAME="ec-$BOARD"
fi
RO_FILE="$BASENAME.RO.flat"

if [ -r $RO_FILE.version ]; then
    NEW_RO_VERSION="$(cat $RO_FILE.version)"
else
    NEW_RO_VERSION="RO version:    N/A"
fi

echo "This script will flash $RO_FILE to the device"
echo "old $CURRENT_RO_VERSION"
echo "new $NEW_RO_VERSION"

user_confirmation

ECTOOL=$ECTOOL RO_FILE=$RO_FILE RO_SIZE=$RO_SIZE make flash

if [ $? -ne 0 ]; then
    echo "update failed"
    exit 1
fi

echo
echo "copying new firmware files"
cp -v $BASENAME.bin $BASENAME.RW.bin /lib/firmware/ni

