#!/bin/bash
# Copyright 2024 Ettus Research, a National Instruments Brand
#
# SPDX-License-Identifier: GPL-3.0-or-later
#

function user_confirmation {
    echo # new line
    read -p "Press \"y\" to continue " -n 1 -r
    echo -e "\n" # two new lines

    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo "aborting"
        exit 1
    fi
}


read -d '' DISCLAIMER <<- EOF
This script updates the microcontroller firmware (RO and RW). The change is
persistent across power cycles for the RO part. Running this script incorrectly can
brick the device. This script may take upto a minute to run successfully.
Ensure that only one X410/X440 device is connected to the host machine at a time.

Pre-requisites:
1. Install openocd package using "sudo apt-get install openocd" on the host machine
2. USB to USB C cable.

Steps:
1. Power off the X410/X440 device by running "shutdown -h now" or "poweroff" command from the linux console. But leave the power connected and turned on.
2. Connect your X410/X440 device using the USB JTAG port to the host machine using a USB cable.
2. Ensure that only one X440/X410 device is connected at a time. No other USRP device should be connected.
3. Run this script.
4. Do not disconnect USB cable while the script is running as it may result in bricking of the device.
5. Power cycle the device after script has completed running successfully.

Updating the microcontroller firmware (RO part) is only required if the Ettus
Research support told you to do so.

EOF

echo "$DISCLAIMER"
user_confirmation


BUILD_DIR=$(pwd)
REV=0

LOWEST_SUPPORTED_REV=5
HIGHEST_SUPPORTED_REV=7


echo "Please enter the motherboard compat_rev number:"
read REV


if [ "$REV" -ge "$LOWEST_SUPPORTED_REV" ] && [ "$REV" -le "$HIGHEST_SUPPORTED_REV" ]; then
    echo "The compat_rev $REV is within the supported range."
else
    echo "Error: The compat_rev number must be between $LOWEST_SUPPORTED_REV and $HIGHEST_SUPPORTED_REV."
    exit 1
fi


FIRMWARE_FILE="ec-titanium-rev$REV.bin"
echo "Hardare Rev: $REV"
echo "Will use firmware file $BUILD_DIR/$FIRMWARE_FILE"

user_confirmation

cp ./$FIRMWARE_FILE ec.bin
echo "Flashing $FIRMWARE_FILE on hardware..."
openocd -c "set BOARD titanium" -c "set BUILD_DIR $BUILD_DIR" -f openocd-flash.cfg

rm ./ec.bin

