#!/bin/bash

# Script to assist NI-KAL clients in installing their kernel component
# Also helps to keep track of NI-KAL client modules installed in the system
#  for reinstallatioin after a kernel upgrade
#
# usage: nikalKernelInstaller.sh [-g] [-o outputFile] [-r|-i|-e] [-n namespace]
#                               [-d installDir] [--linuxrt]
#                               </pathto/client-unversioned.o>
#    Links client-unversioned.o with clientKalInterface.o and installs the
#    resulting module in /lib/modules/`uname -r`/kernel/natinst/<installPath>
#
# -g -o and -f options are internally used by NI-KAL and are "undocumented" by echoUsage
#
# -g        will globally update all kernel modules NI-KAL currently has a
#           record of by recompiling against the kernel and cleaning up any
#           broken links
#
# -o        will just create the resulting output file that would be installed,
#           it is not intended for installation and will not create a record for
#           NI-KAL to keep track of it
#
# -f        requests a fast install to optimize installation time especially on
#           slower systems.  This would suppress calls such as depmod which is
#           useful when installing multiple kernel modules in a row so you can
#           then run depmod or this script with '-g'
#
# --linuxrt Set various options/configurations for doing KAL kernel module work
#           when run from an NI Linux RT target
#
#  version 14.0.0b2
#
#   Copyright 2003-2005,
#  National Instruments Corporation.
#  All Rights reserved.
#
#  originated:  4 May 2004
#

echoUsage()
{
   echo "Usage: $0 [-r|-i|-e] [-d installDir] </pathto/client-unversioned.o>" > /dev/stderr
   echo "option:" > /dev/stderr
   echo " -i, -r, or -e must be provided." > /dev/stderr
   echo " i install    - install the kernel module." > /dev/stderr
   echo " r remove     - remove the kernel module instead of installing it." > /dev/stderr
   echo " e exists     - queries if the given kernel module is installed" > /dev/stderr
   echo "                If no -d argument is given -e will search all modules" > /dev/stderr
   echo "                installed.  Only one module of the same name should" > /dev/stderr
   echo "                ever be installed." > /dev/stderr
   echo " " > /dev/stderr
   echo " d directory  - specify an optional directory to install the kernel" > /dev/stderr
   echo "                module into in the standard location:" > /dev/stderr
   echo "                /lib/modules/\`uname -r\`/kernel/natinst/<installDir>" > /dev/stderr
   echo "                If this is not provided a warning will result" > /dev/stderr
   echo " h help       - prints this usage message" > /dev/stderr
   echo " " > /dev/stderr
   echo "</pathto/client-unversioned.o> is required" > /dev/stderr
   echo "   specifies the installed location of the unversioned binary" > /dev/stderr
   echo "   that needs to link against the clientKalInterface." > /dev/stderr
   echo "   The actual path to client-unversioned.o is only required" > /dev/stderr
   echo "   during installation." > /dev/stderr
}

parseArguments()
{
   clientModuleLocation=""

   while [ "$1" != "" ]
   do
      case $(printf "%s\n" "$1" | awk '{print $1}') in
         -r)
            if [ "$moduleOperation" != "" ]
            then
               echo "-e -r and -i are mutually exclusive and should only be specified once." > /dev/stderr
               echoUsage
               exit $statusFail
            fi
            moduleOperation=remove
            ;;
         -i)
            if [ "$moduleOperation" != "" ]
            then
               echo "-e -r and -i are mutually exclusive and should only be specified once." > /dev/stderr
               echoUsage
               exit $statusFail
            fi
            moduleOperation=install
            ;;
         -e)
            if [ "$moduleOperation" != "" ]
            then
               echo "-e -r and -i are mutually exclusive and should only be specified once." > /dev/stderr
               echoUsage
               exit $statusFail
            fi
            moduleOperation=exists
            ;;
         -g)
            if [ "$moduleOperation" != "" ]
            then
               echo "-e -r -i -g and -o are mutually exclusive and should only be specified once." > /dev/stderr
               echoUsage
               exit $statusFail
            fi
            moduleOperation=globalUpdate
            ;;
         -f)
            fastInstall=1
            ;;
         -h)
            echoUsage
            exit $statusFail
            ;;
         -o)
            if [ "$moduleOperation" != "" ]
            then
               echo "-e -r -i -g and -o are mutually exclusive and should only be specified once." > /dev/stderr
               echoUsage
               exit $statusFail
            fi

            if [ "$1" = "-o" ]
            then
               shift
               moduleOutputFile=$1
            else
               moduleOutputFile=$(printf "%s\n" "$1" | cut -c 2-)
            fi
            moduleOutputFile=${moduleOutputFile}.ko
            moduleOperation=output
            ;;
         -d)
            if [ "$1" = "-d" ]
            then
               shift
               clientInstallPath=$1
            else
               clientInstallPath=$(printf "%s\n" "$1" | cut -c 2-);
            fi
            ;;
         -n)
            if [ "$1" = "-n" ]
            then
               shift
               namespace=$1
            else
               namespace=$(printf "%s\n" "$1" | cut -c 2-)
            fi
            ;;
         --linuxrt)
            linuxrtSettings=1
            fastInstall=1
            ;;
         *)
            if [ "$(printf "%s\n" "$1" | cut -c -1)" = "-" ]
            then
               echo "Unrecognized Option: $1" > /dev/stderr
               echoUsage
               exit $statusFail
            elif [ "$clientModuleLocation" != "" ]
            then
               echo "Received unversioned module location: $clientModuleLocation already. Cannot accept another module name: $1" > /dev/stderr
               echoUsage
               exit $statusFail
            else
               clientModuleLocation=$1
            fi
            ;;
      esac
      shift
   done

   if [ "$moduleOperation" = "globalUpdate" ]
   then
      return $statusSuccess
   fi

   if [ "$clientModuleLocation" = "" ]
   then
      echo "client unversioned module location/name not specified!" > /dev/stderr
      echoUsage
      exit $statusFail
   fi

   if [ "$moduleOperation" = "" ]
   then
      echo "You must choose to install, remove or exists check with -r -i or -e" > /dev/stderr
      echoUsage
      exit $statusFail
   fi

   if [ "$nikalDir" = "" ] && [ "$moduleOperation" != "output" ]
   then
      # output mode can work without NI-KAL installed but it is an internal feature
      echo "NI-KAL is not installed, please install it before running $0" > /dev/stderr
      exit $statusFail
   fi

   if [ "$moduleOperation" = "install" ] || [ "$moduleOperation" = "output" ]
   then
      if [ ! -e $clientModuleLocation ]
      then
         echo "client unversioned module not found!" > /dev/stderr
         echo "checked $clientModuleLocation" > /dev/stderr
         echoUsage
         exit $statusFail
      fi
   fi

   # process client module location information and store in "globals"
   clientModulePath=${clientModuleLocation%\/*}
   if [ "$clientModulePath" = "$clientModuleLocation" ]
   then
      clientModulePath=`pwd`
   fi
   clientModuleUnversioned=$(echo $clientModuleLocation|sed s#^${clientModulePath}\/##)
   clientModuleBase=$(echo $clientModuleUnversioned|sed 's#-unversioned.o$##')
   clientModule=${clientModuleBase}.ko

   # post process clientModulePath to absolute path in case it is a relative path
   if [ "$clientModulePath" != "" ]
   then
      currentPwd=`pwd`
      cd "$clientModulePath"
      clientModulePath=`pwd`
      cd "$currentPwd"
   fi

   if [ "$moduleOperation" = "output" ]
   then
      ldSourceFile=$clientModulePath/$clientModuleUnversioned
   else
      ldSourceFile=$kalClientDB/$clientInstallPath/$clientModuleUnversioned
   fi
}

# depends on global variables
#  nikalDir : nikal's installation directory, default /usr/local/natinst/nikal
#  ldSourceFile : location of unversioned module to version against the kernel
#  clientModuleBase : basename of the client module we are installing
#  clientModule : name of the installed module, nipalk.o/.ko
# arguments
#  $1 : install/target location for the final binary
#       if installing this should be somewhere in /lib/modules/`uname -r`/kernel/natinst/
linkKernelDriver()
{
   ldTargetFile=$1
   if [ "$nikalDir" = "" ]; then
      cd client
      if ! make CLIENT_MODULE:=$ldSourceFile CLIENT_NAME:=$clientModuleBase >/tmp/clientKalMake.log 2>&1; then
         cat /tmp/clientKalMake.log > /dev/stderr
         echo "ERROR: failed to build $clientModule" > /dev/stderr
         return $statusFail
      fi
      cd $OLDPWD
   else
      cd $nikalDir/src/client
      if ! make CLIENT_MODULE:=$ldSourceFile CLIENT_NAME:=$clientModuleBase >/tmp/clientKalMake.log 2>&1; then
         cat /tmp/clientKalMake.log > /dev/stderr
         echo "ERROR: failed to build $clientModule" > /dev/stderr
         return $statusFail
      fi
      cd $OLDPWD
      mv $nikalDir/src/client/$clientModule $ldTargetFile
   fi

   return $statusSuccess
}

# depends on global variables
#  moduleBasePath : /lib/modules/`uname -r`/kernel/natinst
#  clientInstallPath : optional sub directory to install in under moduleBasePath
#  clientModuleBase : basename of the installed module, nipalk
#  clientModule : name of the installed module, nipalk.o
#  clientModuleUnversioned : unversioned name of the module, nipalk-unversioned.o
#  kalClientDB : base directory of the cache where NI-KAL keeps track of client modules
#  clientModulePath : Location of the original client unversioned module
#                     i.e. /usr/local/natinst/nipal/src/objects/nipalk-unversioned.o
installKernelDriver()
{
   source $kernelDir/.config
   if [[ "$linuxrtSettings" == "0" ]] && [[ `file $clientModulePath/$clientModuleUnversioned` == *32-bit* ]] && [[ "$CONFIG_X86_64" == "y" ]] ; then
      echo "INFO: 32-bit module $clientModuleBase not needed for 64-bit kernel."
      return $statusSuccess
   fi


   # look for previously installed components of the same name
   if [ -d "$kalClientDB" ]
   then
      numClientsFound=`find $kalClientDB -name $clientModuleUnversioned | wc -l | sed 's/ //g'`

      if [ "$numClientsFound" -gt 1 ]
      then
         # no good, we shouldn't ever have more then one of the same name installed
         echo "ERROR: multiple kernel modules named $clientModule are already installed!" > /dev/stderr
         echo "ERROR: This should not be allowed and must be fixed!" > /dev/stderr
         return $statusFail
      elif [ "$numClientsFound" -eq 1 ]
      then
         clientFoundName=`find $kalClientDB -name $clientModuleUnversioned`
         clientFoundName=$(echo $clientFoundName|sed s#^${kalClientDB}/##)
         clientFoundInstallPath=$(echo $clientFoundName|sed s#/${clientModuleUnversioned}##)

         # if clientModule was installed previously without -d
         if [ "$clientFoundInstallPath" = "$clientModuleUnversioned" ]
         then
            clientFoundInstallPath=""
         fi

         if [ "$clientFoundInstallPath" != "$clientInstallPath" ]
         then
            if [ -e "$clientFoundName" ]; then
               # component with the same name was previously installed in a different path
               # we don't allow/want that
               echo "ERROR: kernel module named $clientModule already installed in -d $clientFoundInstallPath!" > /dev/stderr
               echo "ERROR: another module with the same name cannot be installed in -d $clientInstallPath." > /dev/stderr
               return $statusFail
            else
               echo "Found stale link in $clientFoundInstallPath." > /dev/stderr
               echo "Removing old entry and adding new entry in $clientInstallPath." > /dev/stderr
               rm -f $clientFoundName
            fi
         fi
      fi
   fi

   if [ "$clientInstallPath" = "" ]
   then
      echo "WARNING: -d was not provided to clientKernelInstall to specify" > /dev/stderr
      echo "WARNING:    a product installation directory. Operation will" > /dev/stderr
      echo "WARNING:    continue but this should be fixed." > /dev/stderr
      echo " " > /dev/stderr
      echo "WARNING: if you are using palmodmgr you should provide the option" > /dev/stderr
      echo "WARNING:   -o linux:dir=<productDir> to palmodmgr." > /dev/stderr
   fi

   if configuredForCurrentKernel
   then
      recursiveUnloadKernelDriver $clientModuleBase
   fi

   mkdir -p $kalClientDB/$clientInstallPath
   ln -sf $clientModulePath/$clientModuleUnversioned \
         $kalClientDB/$clientInstallPath/$clientModuleUnversioned
   mkdir -p $moduleBasePath/$clientInstallPath
   if linkKernelDriver $moduleBasePath/$clientInstallPath/$clientModule; then
      if [ "$fastInstall" = "0" ]
      then
         if configuredForCurrentKernel; then
            if ! /sbin/depmod -a >/tmp/nikalKernelInstallerDepmodOut.txt 2>&1; then
               echo "ERROR: running depmod failed during installKernelDriver" > /dev/stderr
               echo "ERROR:  depmod output follows:" > /dev/stderr
               echo " " > /dev/stderr
               cat /tmp/nikalKernelInstallerDepmodOut.txt > /dev/stderr
               return $statusFail
            fi
         fi
      fi
   else
      echo "ERROR: Run updateNIDrivers after fixing the problem to complete installation." > /dev/stderr
      return $statusFail
   fi

   return $statusSuccess
}

# depends on global variables
#  moduleBasePath : /lib/modules/`uname -r`/kernel/natinst
#  clientInstallPath : optional sub directory to install in under moduleBasePath
#  clientModule : name of the installed module, nipalk.o
#  clientModuleBase : basename of the installed module, nipalk
#  clientModuleUnversioned : unversioned name of the module, nipalk-unversioned.o
#  kalClientDB : base directory of the cache where NI-KAL keeps track of client modules
removeKernelDriver()
{
   if ! existsKernelDriver
   then
      # do nothing if it is not installed
      return $statusSuccess
   fi

   if configuredForCurrentKernel
   then
      recursiveUnloadKernelDriver $clientModuleBase
   fi

   if [ -d "$moduleBasePath/$clientInstallPath" ]; then
      rm -f $moduleBasePath/$clientInstallPath/$clientModule
      ( find $moduleBasePath/$clientInstallPath -type d | sort -r | xargs rmdir ) 2> /dev/null
   fi
   if [ -d "$kalClientDB/$clientInstallPath" ]; then
      rm -f $kalClientDB/$clientInstallPath/$clientModuleUnversioned
      ( find $kalClientDB/$clientInstallPath -type d | sort -r | xargs rmdir ) 2> /dev/null
   fi

   if [ "$fastInstall" -eq 0 ]
   then
      if configuredForCurrentKernel; then
         if ! /sbin/depmod -a >/tmp/nikalKernelInstallerDepmodOut.txt 2>&1; then
            echo "WARNING: running depmod failed during removeKernelDriver" > /dev/stderr
            echo "WARNING:  depmod output follows:" > /dev/stderr
            echo " " > /dev/stderr
            cat /tmp/nikalKernelInstallerDepmodOut.txt > /dev/stderr
         fi
      fi
   fi

   return $statusSuccess
}

# depends on global variables
#  moduleOutputFile : location the target file should be after it is versioned to the kernel
outputKernelDriver()
{
   moduleOutputPath=${moduleOutputFile%\/*}
   clientModule=${moduleOutputFile/#$moduleOutputPath\//}
   clientModuleBase=${clientModule%\.*}

   linkKernelDriver $moduleOutputFile

   return $?
}

globalUpdateKernelDriver()
{
   globalUpdateStatus=$statusSuccess

   if [ -d "$kalClientDB" ]; then
      echo "Updating client modules:"
      for clientModuleFile in `$nikalDir/src/client/processmodule.sh --cmd_sortdependency --kaldbdir=$kalClientDB`
      do
         clientModule=${clientModuleFile#$kalClientDB\/}
         clientInstallPath=${clientModule%\/*}
         clientModuleUnversioned=$(echo $clientModule|sed s#^${clientInstallPath}/## )
         clientModuleBase=$(echo "$clientModuleUnversioned"|sed 's#-unversioned.o##')
         clientModule=${clientModuleBase}.ko
         ldSourceFile=$clientModuleFile

         if configuredForCurrentKernel; then
            recursiveUnloadKernelDriver $clientModuleBase
         fi

         mkdir -p $moduleBasePath/$clientInstallPath
         if linkKernelDriver $moduleBasePath/$clientInstallPath/$clientModule; then
            echo " $clientModule successfully updated."
         else
            echo " $clientModule failed to update."
            globalUpdateStatus=$statusFail
         fi
      done
      rm -Rf /tmp/strippedNIModules
      if [ "$fastInstall" -eq 0 ]
      then
         if configuredForCurrentKernel; then
            if ! /sbin/depmod -a >/tmp/nikalKernelInstallerDepmodOut.txt 2>&1; then
               echo "ERROR: running depmod failed during globalUpdateKernelDriver" > /dev/stderr
               echo "ERROR:  depmod output follows:" > /dev/stderr
               echo " " > /dev/stderr
               cat /tmp/nikalKernelInstallerDepmodOut.txt > /dev/stderr
               globalUpdateStatus=$statusFail
            fi
         fi
      fi
   fi

   return $globalUpdateStatus
}

# depends on global variables
#  moduleBasePath : /lib/modules/`uname -r`/kernel/natinst
#  clientInstallPath : optional sub directory to install in under moduleBasePath
#  clientModule : name of the installed module, nipalk.o
#  clientModuleUnversioned : unversioned name of the module, nipalk-unversioned.o
#  kalClientDB : base directory of the cache where NI-KAL keeps track of client modules
existsKernelDriver()
{
   statusFound=$statusSuccess
   statusNotFound=$statusFail

   if [ ! -d "$kalClientDB" ]
   then
      # cache doesn't exist yet so nothing is installed
      return $statusNotFound
   fi

   # -e without -d will search the entire cache
   if [ "$clientInstallPath" = "" ]
   then
      clientFoundName=`find $kalClientDB -name $clientModuleUnversioned`
      if [ "$clientFoundName" = "" ]
      then
         return $statusNotFound
      else
         clientInstallPath=$(echo "$clientFoundName" | sed s#^${kalClientDB}/##)
         clientInstallPath=$(echo "$clientInstallPath" | sed s#/${clientModuleUnversioned}##)

         # if clientModule was installed previously without -d
         if [ "$clientInstallPath" = "$clientModuleUnversioned" ]
         then
            clientInstallPath=""
         fi
      fi
   fi

   if [ ! -L "$kalClientDB/$clientInstallPath/$clientModuleUnversioned" ] || [ ! -e "$moduleBasePath/$clientInstallPath/$clientModule" ]
   then
      # not in the cache or not installed
      return $statusNotFound
   fi

   return $statusFound
}

default_versioning_env ()
{
   "$@"
}

# global variables
moduleOperation=""
clientInstallPath=""
clientModulePath=""
clientModuleUnversioned=""
clientModule=""
moduleOutputFile=""
ldSourceFile=""
linuxrtSettings=0
versioningEnv=default_versioning_env
# default namespace, NI products shouldn't need to change this
# namespace in KAL has been deprecated. This is left here, but unused
namespace=nNIPAL100
atomicInterfaceSymbol=nNIAPALS100_driverEntry
kalInterfaceSymbol=_driverEntry
clientKalSymbol=_ckalcbInitModule
fastInstall=0
statusSuccess=0
statusFail=1

if [ -e /etc/natinst/nikal/nikal.dir ]; then
   nikalDir=`cat /etc/natinst/nikal/nikal.dir`
   kalClientDB=$nikalDir/etc/clientkdb
else
   # internal use case, from setup.sh buildkernel
   nikalDir=""
   kalClientDB=clientkdb
fi

######## MAIN ######## 

parseArguments $*

# Set up the linux-rt environment
if [[ "$linuxrtSettings" == "1" ]] ; then
   # Source the utility script to get things setup for versioning
   . /usr/local/natinst/tools/versioning_utils.sh

   # Set the stage for doing the actual versioning
   setup_versioning_env

   versioningEnv=versioning_call
fi

if [ -e /etc/natinst/nikal/nikal.dir ]; then
   $versioningEnv . $nikalDir/bin/installerUtility.sh
   $versioningEnv . $nikalDir/src/nikal/Makefile.in
else
   # internal use case, from setup.sh buildkernel
   # source installerUtility.sh for testKernelDriverLoaded and recursiveUnloadKernelDriver
   $versioningEnv . ../bin/installerUtility.sh
   $versioningEnv . ./Makefile.in
fi

# MODPATH and KERNELDIRECTORY come from Makefile.in which was created from configure
moduleBasePath=$MODPATH
kernelDir=$KERNELDIRECTORY

if ! $versioningEnv configuredForCurrentKernel; then
   echo "warning: configured for kernel version $KERNELTARGET"
   echo "warning: which is not the currently running kernel"
fi

# must be root except when running output
if [ `id -ur` -ne 0 ] && [ "$moduleOperation" != "output" ]
then
   echo "Please run NI-KAL client kernel module installer as root" > /dev/stderr
   exit $statusFail
fi

# Execute the requested operation
$versioningEnv ${moduleOperation}KernelDriver

if [[ "$linuxrtSettings" == "1" ]] ; then
   # Cleanup the environment
   cleanup_versioning_env
fi

exit $?

