#!/bin/bash
# Copyright 2013,
# National Instruments Corporation.
# All rights reserved.

declare -A shorten
declare -iA symcount
declare -A undefined
declare -A exported

declare -a modules
declare -A modules_inv

isLinkBroken () {
	readlink -q $1 >/dev/null
}

removeBrokenLink () {
	rm -f "$1" || return $?
	#if directory containing link is empty, remove directory containing link
	rmdir -f "$(dirname $1)" >&/dev/null
	return 0
}

# loadShortenSymFile () - move symbols file contents to shorten
# $1 - symbols file, containing exported function symbols for processed modules (shortsyms.txt)
loadShortenSymFile () {
	if [[ -e $1 ]]; then
		while read sym modexport; do
			local modname="${modexport%%_exported*}"

			if [[ $modname != $modexport ]]; then
				shorten["${modname} $sym"]="$modexport"
				(( symcount["$modname"]++ ));
			fi
		done <$1
	fi
}

# printExportedSyms () - returns exported symbols for module
# $1 - module binary
printExportedSyms () {
	strings $1 | sed -ne \
		'/nNIExport_beginExportedSymbolsArray/ {
			n
			:a
			/nNIExport_endExportedSymbolsArray/q
			p
			n
			ba
		}'
}

# printPCITable () - returns hw IDs for module
# $1 - module binary
printPCITable () {
	strings $1 | sed -ne \
		'/nNIKAL200_beginPCIIDTable/ {
			n
			:a
			/nNIKAL200_endPCIIDTable/q
			p
			n
			ba
		}'
}

# updateSymbolRenameFile () - updates symbols file with exported functions of given module
# $1 - symbols file (shortsyms.txt)
# $2 - module binary
updateSymbolRenameFile () {
	# we either call this function or we eliminate duplicates
	loadShortenSymFile $1

	local maxSymbolNameLength=55
	local modname="$2"
	modname="${modname%%-unversioned.o}"
	modname="${modname##*/}"

	# Abort if module binary doesn't exist
	test -e $2 || return 1

	# Abort if output file cannot be created/written
	touch $1 || return 1

	declare -i i=${symcount["$modName"]}
	for symbol in $(printExportedSyms $2); do
		if (( ${#symbol} > $maxSymbolNameLength )); then
			if ! [[ ${shorten["$modname $symbol"]+x} ]]; then
				shorten["$modname $symbol"]=
				printf "%s %s_exported%d\n" "$symbol" "$modname" "$i"
				(( i++ ))
			fi
		fi
	done >> $1
	# prevent dupes explicitly with uniq if loadShortenSymFile isn't called

	# This function always returns success if it reaches this point
	return 0
}

# tsortFunc () - sorts and displays all modules by given dependency array (topological sort),
# while modifying <modules_inv> associative array
# $1 - array with line-by-line dependencies
# the complexity of this function is O(n^2) reorderings and O(n) rescans of the input,
# but we are dealing with tens of dependencies, not thousands, so it's fine
tsortFunc () {
	local -i notDone=1
	while (( $notDone )); do
		(( notDone = 0 ))
		while read expSymModule module; do
			if [[ -n "$module" ]]; then
				# $module depends on $expSymModule
				if (( ${modules_inv[$module]} <=  ${modules_inv[$expSymModule]} )); then
					(( modules_inv[$module]= ${modules_inv[$expSymModule]} + 1 ))
					(( notDone = 1 ))
				fi
			fi
		done <<< "$1"
	done

	echo "$( for module in ${!modules_inv[@]}; do
		printf '%03d %s\n' ${modules_inv[$module]} $module
	done | sort | awk '{print $2}')"
}

# sortModules () - sorts all modules by dependency (topological sort)
# uses global variables <undefined> and <exported> passed from sortDependency ()
# 	to determine module dependencies
sortModules () {
	local dependencies=$(
		for module in "${!undefined[@]}"; do
			for symbol in ${undefined[${module}]}; do
				local expSymModule="${exported[${symbol}]}"
				if [[ -n $expSymModule ]]; then
					# $module depends on $expSymModule, print dependency
					printf '%s %s\n' "$expSymModule" "$module"
				fi
			done
		done | sort | uniq)

	tsortFunc "$dependencies"
}

# sortDependency () - sorts all modules in given directory by dependency
# $1 - directory containing all modules
sortDependency ()
{
	while IFS= read -r -d $'\0' l; do
		isLinkBroken "$l" || { removeBrokenLink "$l" && continue; }

		modules+=("$l")

		local module="$l"
		undefined[${module}]=`nm -u $module | awk '{print \$2}'`

		for expsym in $(printExportedSyms $module); do
			exported[${expsym}]=$module
		done
	done < <(find $1 -type l -print0)

	for i in ${!modules[@]}; do
		modules_inv["${modules[$i]}"]=$i
	done

	sortModules
}

# printInterface () - generate wrapper from module based on given symbols file
# $1 - symbols file
# $2 - module binary
printInterface ()
{
	loadShortenSymFile $1

	local readonly moduleFile="$2"

	local exportedSyms="$(printExportedSyms $moduleFile)"
	local pciTable="$(printPCITable $moduleFile)"

	readonly -A interfaceTypeMap=(
	[nNIAPAL200_driverEntry]=atomicInterface
	[nNIPAL100_driverEntry]=kalInterface
	[nNIPAL100_ckalcbInitModule]=clientKalInterface)
	# determine client interface interfaceType
	interfaceType=
	while read line; do
		interfaceType=${interfaceTypeMap[$line]}
		[[ -z $interfaceType ]] || break
	done <<< "$exportedSyms"

	if  [[ -z $interfaceType ]]; then
		echo "Unknown Interface Type !!!"
		exit 1
	fi

	modname=$(basename ${moduleFile%-unversioned\.o})

	#
	# Now generate module-interface.c
	#

	cat << "EOF"

#undef __NO_VERSION__
#include <linux/version.h>

#ifdef nNIKAL160_kConfig
   #include <linux/config.h>
#endif
#include <linux/string.h>
#include <linux/errno.h>
#ifdef MODULE
   #if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
      #define MODVERSIONS
   #endif

   #ifdef MODVERSIONS
      #include <config/modversions.h>
   #endif

   #include <linux/module.h>
#endif

#include "nikal.h"

#ifdef MODULE_LICENSE
   #ifdef nNIKAL100_kCopyrightString
      MODULE_LICENSE(nNIKAL100_kCopyrightString);
   #endif
#endif

#ifdef MODULE_AUTHOR
   #ifdef nNIKAL100_kAuthorString
      MODULE_AUTHOR(nNIKAL100_kAuthorString);
   #endif
#endif

#ifdef MODULE_DESCRIPTION
   #ifdef nNIKAL100_kDescriptionString
      MODULE_DESCRIPTION(nNIKAL100_kDescriptionString);
   #endif
#endif

EOF
	if [[ $interfaceType = "kalInterface" ]]; then
		cat << "EOF"
static struct nNIKAL100_tDriver kalDriver;
nNIKAL100_cc int nNIPAL100_driverEntry(struct nNIKAL100_tDriver *driver);

int init_module(void)
{
#ifdef __i386__
   /** We don't support machine configuration with physical memory located
    *  above the 4GB physical address boundary on 32bit systems.
    *  This typically occurs for systems that have more than 3GB memory.  The
    *  BIOS needs to reserve some physical address space for device mapping,
    *  so it puts any additional memory beyond 3GB above the 4GB physical
    *  address space.
    *  Below, we check if addressableMemory overflows a 32 bit value.
    *  On 32bit systems this occurs when there is more than 4GB of addressable memory.
    **/

   // PAL is the only one that use this. Fail 32bit PAL load if amount of addressable memory is > 4G, since PAL can't handle it.
   if (nNIKAL200_isAddressableMemOver4G())
   {
      nNIKAL100_printToDebugger("[nipal] More than 4GB of addressable memory detected.\n");
      nNIKAL100_printToDebugger("[nipal] This configuration is not supported.  Check the release notes for more information.\n");
      return -ENOSYS;
   }
#endif

   /* We need to initialize the client first before registering our driver to the
    * kernel. This is to guarantee that we're ready to receive connection from
    * user mode right after we register to the kernel. */
   if (nNIPAL100_driverEntry(&kalDriver))
      return -ENODEV;
   kalDriver.module = THIS_MODULE;

   /* on fail registerDriver will call unregisterDriver which will
    * subsequently call driver->unload for us */
   return nNIKAL100_registerDriver(&kalDriver);
}

void cleanup_module(void)
{
   /* unregisterDriver also calls driver->unlod for us */
   nNIKAL100_unregisterDriver(&kalDriver);
}

EOF
	elif [[ $interfaceType = "clientKalInterface" ]]; then
		cat << "EOF"
nNIKAL100_cc int nNIPAL100_ckalcbInitModule(void* modulePtr, const char* name);
nNIKAL100_cc void nNIPAL100_ckalcbCleanupModule(void* modulePtr);

int init_module(void)
{
   return nNIPAL100_ckalcbInitModule(THIS_MODULE,(THIS_MODULE)->name);
}

void cleanup_module(void)
{
   nNIPAL100_ckalcbCleanupModule(THIS_MODULE);
}

EOF
	elif [[ $interfaceType = "atomicInterface" ]]; then
		cat << "EOF"
#include <linux/pci.h>

nNIKAL100_cc int nNIAPAL200_driverEntry(struct nNIKAL100_tDriver *driver);

static struct nNIKAL100_tDriver atomicDriver;
static void* nativeDriverRegistration;

/* native driver registration functions. These functions are auto-generated */
void *registerNativeDriver(nNIKAL100_tDriver*);
void unregisterNativeDriver(void*);

int init_module(void)
{
   atomicDriver.module = THIS_MODULE;

   if (nNIAPAL200_driverEntry(&atomicDriver))
      return -ENODEV;

   nativeDriverRegistration = registerNativeDriver(&atomicDriver);

   if (nativeDriverRegistration == NULL)
   {
      if (atomicDriver.unload)
         atomicDriver.unload(&atomicDriver);
      return -ENODEV;
   }

   return 0;
}

void cleanup_module(void)
{
   unregisterNativeDriver(nativeDriverRegistration);

   if (atomicDriver.unload)
      atomicDriver.unload(&atomicDriver);
}

EOF

		#
		# generate hw ID table and native driver registration
		#
		if  [[ -n "$pciTable" ]]; then
			cat << "EOF"
static struct pci_device_id nipciidtable[] = {
EOF

			while read line; do
				printf "   {"
				i=0
				for id in $line; do
					if [[ $id = "none" ]]; then
						if (( i == 4 || i == 5 )); then
							# these are pci class spec. use '0' instead of None
							printf "0, "
						else
							printf "PCI_ANY_ID, "
						fi
					else
						printf "%s, " "$id"
					fi
					(( i++ ))
				done
				printf "(unsigned long)&atomicDriver},\n"
			done <<< "$pciTable"

			cat << EOF

   { },
};
MODULE_DEVICE_TABLE(pci, nipciidtable);

static struct pci_driver niPciDriver = {
   .name = "$modname",
   .id_table = nipciidtable,
};

/* native driver registration is no-op */
void *registerNativeDriver(nNIKAL100_tDriver* ignore)
{
   return nNIKAL200_registerPCIDriver(&niPciDriver);
}
void unregisterNativeDriver(void* data)
{
   nNIKAL200_unregisterPCIDriver(data);
}

EOF
		else
			# no PCI Table
			cat << "EOF"
/* native driver registration is no-op */
void *registerNativeDriver(struct nNIKAL100_tDriver* ignore)
{
   static int some_non_null_storage_ignoreme;
   return &some_non_null_storage_ignoreme;
}
void unregisterNativeDriver(void* ignore) { }

EOF

		fi #if [[ -n $pciTable ]]; then
	fi #elif [[ $interfaceType = "atomicInterface" ]]; then

	#
	# generate symbol exports
	#
	cat << "EOF"
#ifdef EXPORT_SYMBOL_NOVERS
#define nNIKAL100_mExportSymbol EXPORT_SYMBOL_NOVERS
#else
#define nNIKAL100_mExportSymbol EXPORT_SYMBOL
#endif
EOF

	#
	# generate export sym section
	#
	i=0
	for sym in $exportedSyms; do
		if ! [[ $sym =~ _(driverEntry|ckalcbInitModule|ckalcbCleanupModule)$ ]]
		then
			if [[ ${shorten["$modname $sym"]+x} ]]; then
				sym="${shorten[$modname $sym]}"
			fi
			printf 'extern int %s;\nnNIKAL100_mExportSymbol(%s);\n' "$sym" "$sym"
		fi
	done
}

parseOptHelp()
{
	cat << "EOF"
Usage: processmodule.sh --cmd_sortdependency --kaldbdir=DIRECTORY
       processmodule.sh --cmd_updatesystemrename --symfile=FILE
       processmodule.sh --cmd_geninterface --symfile=FILE --module=MODULE

Helper script for kernel module installation for National Instruments

Options:
  -h, --help            show this help message and exit
  --cmd_sortdependency  print out the modules sorted in dependency, for all
                        modules in kaldbdir
  --cmd_updatesymrename
                        update the file that contains the list of symbols that
                        needs renaming
  --cmd_geninterface    generate the interface file and print to stdout
  --kaldbdir=KALDBDIR   ni-kal client db directory
  --symfile=SYMFILE     the symbol file to use
  --module=MODULE       the module path
EOF
}

# parse command options and run necessary use case
# $1 - all options passed to this script
parseOptAndRun()
{
	local -i v_sortdependency=0
	local -i v_updatesymrename=0
	local -i v_geninterface=0
	local v_kaldbdir=
	local v_symfile=
	local v_module=

	while test $# -gt 0; do
		case "$1" in
			-h | --help)
				parseOptHelp
				exit
				;;
			--cmd_sortdependency)
				v_sortdependency=1
				;;
			--cmd_updatesymrename)
				v_updatesymrename=1
				;;
			--cmd_geninterface)
				v_geninterface=1
				;;
			--kaldbdir=*)
				v_kaldbdir=${1#*=}
				;;
			--symfile=*)
				v_symfile=${1#*=}
				;;
			--module=*)
				v_module=${1#*=}
				;;
		esac
		shift
	done

	if (( $v_sortdependency == 0 && $v_updatesymrename == 0 && $v_geninterface == 0 )); then
		echo -e "\nYou must specify only one of the commands\n"
		parseOptHelp
		exit 1
	fi

	if (( $v_sortdependency == 1 )); then
		if [[ -n $v_kaldbdir ]]; then
			sortDependency $v_kaldbdir
			exit
		else
			echo -e "\nNo value given to --kaldbdir\n"
		fi
	elif (( $v_updatesymrename == 1 )); then
		if [[ -n $v_symfile && -n $v_module ]]; then
			updateSymbolRenameFile $v_symfile $v_module
			exit
		else
			echo -e "\nNo value given to either --symfile or --module\n"
		fi
	elif (( $v_geninterface == 1 )); then
		if [[ -n $v_symfile && -n $v_module ]]; then
			printInterface $v_symfile $v_module
			exit
		else
			echo -e "\nNo value given to either --symfile or --module\n"
		fi
	fi

	parseOptHelp
	exit 1
}

# parse command options and run necessary use case
parseOptAndRun $@

