#!/bin/bash

# Script to assist NI-KAL clients in installing their kernel component
# Also helps to keep track of NI-KAL client modules installed in the system
#  for reinstallation after a kernel upgrade
#
# usage: nikalKernelInstaller.sh [-g] [-o outputFile] [-r|-i|-e] [-n namespace]
#                               [-d installDir] [--linuxrt]
#                               </pathto/client-unversioned.o>
#    Links client-unversioned.o with clientKalInterface.o and installs the
#    resulting module in /lib/modules/$(uname -r)/kernel/natinst/<installPath>
#
# -g -o and -f options are internally used by NI-KAL and are "undocumented" by echoUsage
#
# -g        will globally update all kernel modules NI-KAL currently has a
#           record of by recompiling against the kernel and cleaning up any
#           broken links
#
# -o        will just create the resulting output file that would be installed,
#           it is not intended for installation and will not create a record for
#           NI-KAL to keep track of it
#
# -f --fast requests a fast install to optimize installation time especially on
#           slower systems.  This would suppress depmod during module install
#           during global update checks if there have been any KAL or KAL-client
#           changes before starting module build, and if there have been no
#           changes skips module build entirely
#
# --build   Force building the specified client-unversioned.o at install time
# --linuxrt (deprecated) Set various options/configurations for doing KAL kernel module work
#           when run from an NI Linux RT target.  It is unnecessary to set this flag anymore.
#
#  version 17.5.1f0
#
#  (C) Copyright 2003-2015,
#  National Instruments Corporation.
#  All Rights reserved.
#

# readlink "$0" can return a relative symlink
# in that case, we need to tread carefully when retrieving the directory
# for the target of the symlink
# __TMPSELF__ is the path of the file (if not a symlink), or the target
# of a symlink, which may be a relative or absolute path
__TMPSELF__=$(if [ -L "$0" ]; then readlink "$0"; else echo "$0"; fi;)
__SELF__=$(basename ${__TMPSELF__})
# In case a symlink was executed, and the symlink target is a relative
# path, we change to the directory that the path is relative to
# the directory containing the symlink), then change to the directory
# containing the symlink target.
__DIR__="$(cd "$(dirname "$0")" && cd "$(dirname "${__TMPSELF__}")" && pwd)"

# Configure logging
LOG_MSG_TAG="nikal"

# Get useful functions to help with install
if [[ -e /usr/share/nikal/installerUtility.sh ]]; then
   source /usr/share/nikal/installerUtility.sh
elif [[ -e /usr/local/natinst/nikal/bin/installerUtility.sh ]]; then
   source /usr/local/natinst/nikal/bin/installerUtility.sh
elif [[ -e "${__DIR__}/../../bin/installerUtility.sh" ]]; then
   source "${__DIR__}/../../bin/installerUtility.sh"
elif [[ -e "${__DIR__}/../bin/installerUtility.sh" ]]; then
   source "${__DIR__}/../bin/installerUtility.sh"
elif [[ -e "${__DIR__}/installerUtility.sh" ]]; then
   source "${__DIR__}/installerUtility.sh"
else
   echo "Error opening the NI-KAL driver utility library." >&2
   exit 1
fi

function echoUsage() {
   info "Usage: $0 [-r|-i|-e] [-d installDir] </pathto/client-unversioned.o>"
   info "option:"
   info " -i, -r, or -e must be provided."
   info " i install    - install the kernel module."
   info " r remove     - remove the kernel module instead of installing it."
   info " e exists     - queries if the given kernel module is installed"
   info "                If no -d argument is given -e will search all modules"
   info "                installed.  Only one module of the same name should"
   info "                ever be installed."
   info " "
   info " d directory  - specify an optional directory to install the kernel"
   info "                module into in the standard location:"
   info "                /lib/modules/\$\(uname -r\)/kernel/natinst/<installDir>"
   info "                If this is not provided a warning will result"
   info " h help       - prints this usage message"
   info " "
   info "</pathto/client-unversioned.o> is required"
   info "   specifies the installed location of the unversioned binary"
   info "   that needs to link against the clientKalInterface."
   info "   The actual path to client-unversioned.o is only required"
   info "   during installation."
}

function parseArguments() {
   clientModuleLocation=""

   while [ "$1" != "" ]; do
      case $(printf "%s\n" "$1" | awk '{print $1}') in
         -r)
            if [ "$moduleOperation" != "" ]; then
               error "Invalid option: -e -r and -i are mutually exclusive and should only be specified once."
               echoUsage
               return 1
            fi
            moduleOperation=remove
            ;;
         -i)
            if [ "$moduleOperation" != "" ]; then
               error "Invalid option: -e -r and -i are mutually exclusive and should only be specified once."
               echoUsage
               return 1
            fi
            moduleOperation=install
            ;;
         -e)
            if [ "$moduleOperation" != "" ]; then
               error "Invalid option: -e -r and -i are mutually exclusive and should only be specified once."
               echoUsage
               return 1
            fi
            moduleOperation=exists
            ;;
         -g)
            if [ "$moduleOperation" != "" ]; then
               error "Invalid option: -e -r -i -g and -o are mutually exclusive and should only be specified once."
               echoUsage
               return 1
            fi
            moduleOperation=globalUpdate
            ;;
         -f|--fast)
            fastInstall=0
            ;;
         --prompt)
            promptUser=0
            ;;
         -h)
            echoUsage
            return 1
            ;;
         -o)
            if [ "$moduleOperation" != "" ]; then
               error "Invalid option: -e -r -i -g and -o are mutually exclusive and should only be specified once."
               echoUsage
               return 1
            fi

            if [ "$1" = "-o" ]; then
               shift
               moduleOutputFile=$1
            else
               moduleOutputFile=$(printf "%s\n" "$1" | cut -c 2-)
            fi
            moduleOutputFile=${moduleOutputFile}.ko
            moduleOperation=output
            ;;
         -d)
            if [ "$1" = "-d" ]; then
               shift
               clientInstallPath=$1
            else
               clientInstallPath=$(printf "%s\n" "$1" | cut -c 2-);
            fi
            ;;
         -n)
            if [ "$1" = "-n" ]; then
               shift
               namespace=$1
            else
               namespace=$(printf "%s\n" "$1" | cut -c 2-)
            fi
            ;;
         --linuxrt)
            # deprecated
            ;;
         --build)
            buildDuringInstall=0
            ;;
         *)
            if [ "$(printf "%s\n" "$1" | cut -c -1)" = "-" ]; then
               error "Unrecognized option: $1"
               echoUsage
               return 1
            elif [ "$clientModuleLocation" != "" ]; then
               error "Invalid option: Unversioned module $clientModuleLocation already specified. Cannot accept another module name: $1"
               echoUsage
               return 1
            else
               clientModuleLocation=$1
            fi
            ;;
      esac
      shift
   done

   if [ "$moduleOperation" = "globalUpdate" ]; then
      return 0
   fi

   if [ -z "$clientModuleLocation" ]; then
      error "Missing required option: client unversioned module location/name not specified!"
      echoUsage
      return 1
   fi

   if [ -z "$moduleOperation" ]; then
      error "Missing required option: must specify install, remove or exists check with -r -i or -e"
      echoUsage
      return 1
   fi

   if [ -z "$nikalDir" ] && [ "$moduleOperation" != "output" ]; then
      # output mode can work without NI-KAL installed but it is an internal feature
      error "NI-KAL is not installed, please install it before running $0"
      return 1
   fi

   if [ "$moduleOperation" = "install" ] || [ "$moduleOperation" = "output" ]; then
      if [ ! -e $clientModuleLocation ]; then
         error "Invalid option: The specified unversioned client module $clientModuleLocation does not exist"
         echoUsage
         return 1
      fi
   fi

   # process client module location information and store in "globals"
   clientModulePath=${clientModuleLocation%\/*}
   if [ "$clientModulePath" = "$clientModuleLocation" ]; then
      clientModulePath=$(pwd)
   fi
   clientModuleUnversioned=$(echo $clientModuleLocation|sed s#^${clientModulePath}\/##)
   clientModuleBase=$(echo $clientModuleUnversioned|sed 's#-unversioned.o$##')
   clientModule=${clientModuleBase}.ko

   # post process clientModulePath to absolute path in case it is a relative path
   if [ "$clientModulePath" != "" ]; then
      (cd "$clientModulePath" && clientModulePath=$(pwd))
   fi

   if [ "$moduleOperation" = "output" ]; then
      ldSourceFile=$clientModulePath/$clientModuleUnversioned
   else
      ldSourceFile=$nikalClientDB/$clientInstallPath/$clientModuleUnversioned
   fi
}

# depends on global variables
#  Returns: zero if kernel is 64-bit
#           nonzero for 32-bit or error while determining kernel bitness
function isKernel64bit() {
   if ! nikalGetConfiguredKernelDir; then
      return 1
   fi

   if [[ -r "${kernelDir}/.config" ]]; then
      (source ${kernelDir}/.config && test "$CONFIG_X86_64" == "y")
      return $?
   else
      return 1
   fi
}

# depends on global variables
#  clientInstallPath : optional sub directory to install in under moduleBasePath
#  clientModule : name of the installed module, nipalk.o
#  clientModuleUnversioned : unversioned name of the module, nipalk-unversioned.o
#  nikalClientDB : base directory of the cache where NI-KAL keeps track of client modules
#  clientModulePath : Location of the original client unversioned module
#                     i.e. /usr/local/natinst/nipal/src/objects/nipalk-unversioned.o
function registerKernelDriver() {
   if ! requirecommands find wc sed mkdir ln; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   # Skip modules that are not compatible with the current kernel architecture
   # Compatibility support for products that, in multiarch scenarios, also install their 32-bit kernel modules on 64-bit systems
   if isKernel64bit && requirecommands file >/dev/null && [[ $(file $clientModulePath/$clientModuleUnversioned) == *32-bit* ]] ; then
      info "Skipping $clientModuleBase: 32-bit module not needed for 64-bit kernel."
      return 0
   fi

   # look for previously registered components of the same name
   if [ -d "$nikalClientDB" ]; then
      numClientsFound=$(find $nikalClientDB -name $clientModuleUnversioned | wc -l | sed 's/ //g')

      if [ "$numClientsFound" -gt 1 ]; then
         # no good, we shouldn't ever have more then one of the same name installed
         error "Multiple kernel modules named $clientModule are already installed!"
         error "This should not be allowed and must be fixed!"
         return 1
      elif [ "$numClientsFound" = "1" ]; then
         clientFoundName=$(find $nikalClientDB -name $clientModuleUnversioned)
         clientFoundName=$(echo $clientFoundName|sed s#^${nikalClientDB}/##)
         clientFoundInstallPath=$(echo $clientFoundName|sed s#/${clientModuleUnversioned}##)

         # if clientModule was installed previously without -d
         if [ "$clientFoundInstallPath" = "$clientModuleUnversioned" ]; then
            clientFoundInstallPath=""
         fi

         if [ "$clientFoundInstallPath" != "$clientInstallPath" ]; then
            if [ -r "$clientFoundName" ]; then
               # component with the same name was previously installed in a different path
               # we don't allow/want that
               error "Kernel module named $clientModule already installed in -d $clientFoundInstallPath!"
               error "Another module with the same name cannot be installed in -d $clientInstallPath."
               return 1
            else
               warning "Found stale link in $clientFoundInstallPath."
               warning "Removing old entry and adding new entry in $clientInstallPath."
               rm -f $clientFoundName
            fi
         fi
      fi
   fi

   if [ -z "$clientInstallPath" ]; then
      warning "The \"-d\" option was not provided to specify a product installation directory."
      warning "Operation will continue but this should be fixed."
      warning " "
      warning "If you are using palmodmgr you should provide the option -o linux:dir=<productDir> to palmodmgr."
   fi

   mkdir -p "$nikalClientDB/$clientInstallPath"
   ln -sf "$clientModulePath/$clientModuleUnversioned" "$nikalClientDB/$clientInstallPath/$clientModuleUnversioned"
   return 0
}

# depends on global variables
#  clientInstallPath : optional sub directory to install in under moduleBasePath
#  clientModuleBase : basename of the installed module, nipalk
#  clientModule : name of the installed module, nipalk.o
#  clientModuleUnversioned : unversioned name of the module, nipalk-unversioned.o
#  nikalClientDB : base directory of the cache where NI-KAL keeps track of client modules
#  clientModulePath : Location of the original client unversioned module
#                     i.e. /usr/local/natinst/nipal/src/objects/nipalk-unversioned.o
#  buildDuringInstall : determines whether the module build steps should be invoked
#                       0 = enabled, 1 = disabled
#  kernelName : if buildDuringInstall was set, and fastInstall is not, then depmod
#               will be run for the kernel $kernelName
# Requires a versioning-capable environment
function installKernelDriver() {
   if ! registerKernelDriver ; then
      error "An error occurred while registering ${clientModuleBase} for kernel module build."
      return 1
   fi

   if [[ "$buildDuringInstall" = "1" ]]; then
      return 0
   fi

   if ! requirecommands mkdir; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   if ! nikalGetConfiguredKernelModuleBasePath; then
      return 1
   fi

   mkdir -p $moduleBasePath/$clientInstallPath
   nikalBuildKernelModule $clientModuleBase $ldSourceFile
   local moduleBuildResult=$?
   # Error
   if [[ ${moduleBuildResult} = "1" ]]; then
      error "Run updateNIDrivers after fixing the problem to complete installation."
      return 1
   fi

   # Module built successfully
   if [[ "${moduleBuildResult}" = "0" ]] && configuredForCurrentKernel; then
      if [[ "${fastInstall}" = "1" ]]; then
         nikalUpdateKernelDepmod
      fi
      recursiveUnloadKernelDriver "${clientModuleBase}"
      reloadKernelDrivers "${nikalUnloadedModules}"

      # Trigger udev to rematch unmatched devices to kernel modules
      # this should cause newly-added drivers for hardware in the system
      # to get loaded and attach to their devices
      if requirecommands udevadm; then
         # Tell udevd to re-read the kernel module indexes
         udevadm control -R
         # Make sure udevd has processed all outstanding events
         udevadm settle
         # Replay (unattached) device events as if they were just coldplugged
         udevadm trigger
      fi
   fi

   return 0
}

# depends on global variables
#  clientInstallPath : optional sub directory to install in under moduleBasePath
#  clientModule : name of the installed module, nipalk.ko
#  clientModuleBase : basename of the installed module, nipalk
#  clientModuleUnversioned : unversioned name of the module, nipalk-unversioned.o
#  nikalClientDB : base directory of the cache where NI-KAL keeps track of client modules
# Requires a versioning-capable environment
function removeKernelDriver() {
   if ! requirecommands find sort xargs rmdir; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   # Clean up all installed, configured kernels
   for moddir in /lib/modules/*/kernel/natinst/"${clientInstallPath}"/; do
      rm -f "${moddir}/${clientModule}"
      rmdir -p "${moddir}" > /dev/null 2>&1
   done

   # Remove client module registration
   if [ -d "${nikalClientDB}/${clientInstallPath}" ]; then
      rm -f "${nikalClientDB}/${clientInstallPath}/${clientModuleUnversioned}"
      find "${nikalClientDB}/${clientInstallPath}/" -type d -depth -exec rmdir '{}' \; > /dev/null 2>&1
   fi

   return 0
}

# depends on global variables
#  moduleOutputFile : location the target file should be after it is versioned to the kernel
# Requires a versioning-capable environment
function outputKernelDriver() {
   moduleOutputPath=${moduleOutputFile%\/*}
   clientModule=${moduleOutputFile/#$moduleOutputPath\//}
   clientModuleBase=${clientModule%\.*}

   nikalBuildKernelModule $clientModuleBase $ldSourceFile

   return $?
}

function promptForReboot() {
   local rebootConfirm=
   while :
   do
      echo -n "Would you like to reboot now? [yes|no] "
      read rebootConfirm
      if [ "$rebootConfirm" != "yes" ] && [ "$rebootConfirm" != "no" ]; then
         echo "Please enter yes or no."
      else
         break
      fi
   done

   if [ "$rebootConfirm" = "yes" ]; then
      info "rebooting"
      reboot
      exit 0
   else
      info "Please reboot manually before attempting to use your NI drivers and products."
   fi
}

# Requires a versioning-capable environment
function globalUpdateKernelDriver() {
   if ! requirecommands mkdir; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   local -a modulesRebuilt=()
   local moduleBuildResult=0

   # As a performance enhancement if fastInstall was set,
   # we'll skip building modules that are already up-to-date
   if [[ "$fastInstall" = "0" ]]; then
      forceRebuild=1
   fi

   nikalBuildKernelModule nikal
   moduleBuildResult=$?
   if [[ "${moduleBuildResult}" = "0" ]]; then
      modulesRebuilt+=("nikal")
   elif [[ "${moduleBuildResult}" = "1" ]]; then
      error " NI-KAL update failed."
      error " make of nikal kernel module failed, not installing kernel module."
      error " updateNIDrivers should be called again after fixing the problem."
      return 1
   fi

   # If there's no nikalClientDB directory, then there are no client modules to build
   if [[ ! -d "$nikalClientDB" ]]; then
      info "No client modules registered."
      return 0
   fi

   for clientModuleFile in $($nikalDir/src/client/processmodule.sh --cmd_sortdependency --kaldbdir=$nikalClientDB); do
      clientModule=${clientModuleFile#$nikalClientDB\/}
      clientInstallPath=${clientModule%\/*}
      clientModuleUnversioned=$(echo $clientModule|sed s#^${clientInstallPath}/## )
      clientModuleBase=$(echo "$clientModuleUnversioned"|sed 's#-unversioned.o##')
      clientModule=${clientModuleBase}.ko
      ldSourceFile=$clientModuleFile

      if ! nikalGetConfiguredKernelModuleBasePath; then
         return 1
      fi

      mkdir -p $moduleBasePath/$clientInstallPath

      nikalBuildKernelModule $clientModuleBase $ldSourceFile
      moduleBuildResult=$?
      if [[ "${moduleBuildResult}" = "0" ]]; then
         modulesRebuilt+=("$clientModule")
      elif [[ "${moduleBuildResult}" = "1" ]]; then
         return 1
      fi
   done

   # If no modules rebuilt, then there's nothing left to do
   if [[ "${#modulesRebuilt[@]}" = "0" ]]; then
      return 0
   fi

   if configuredForCurrentKernel; then
      if [ "$fastInstall" = "1" ]; then
         nikalUpdateKernelDepmod
      fi

      # Associative array to remove duplicates
      local -A unloadedModules=()
      for rebuiltModule in "${modulesRebuilt[@]}"; do
         recursiveUnloadKernelDriver $rebuiltModule
         for mod in "${nikalUnloadedModules[@]}"; do
            unloadedModules[$mod]=1
         done
      done
      # Recover the keys of the associative array into a regular array
      nikalUnloadedModules=(${!unloadedModules[@]})
      reloadKernelDrivers "${nikalUnloadedModules}"

      # Trigger udev to rematch unmatched devices to kernel modules
      # this should cause newly-added drivers for hardware in the system
      # to get loaded and attach to their devices
      if requirecommands udevadm; then
         udevadm trigger
      fi

      if [ "${promptUser}" = "0" ]; then
         promptForReboot
      else
         info "Rebooting is recommended to ensure that all new and updated"
         info "National Instruments driver modules have been successfully updated."
      fi
   else
      info "National Instruments kernel drivers have been successfully installed."
      info "You can now use your NI products when you boot into kernel ${KERNELTARGET}."
   fi

   return 0
}

# depends on global variables
#  clientInstallPath : optional sub directory to install in under moduleBasePath
#  clientModule : name of the installed module, nipalk.o
#  clientModuleUnversioned : unversioned name of the module, nipalk-unversioned.o
#  nikalClientDB : base directory of the cache where NI-KAL keeps track of client modules
# Requires a versioning-capable environment
function existsKernelDriver() {
   if ! requirecommands find sed; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   statusFound=0
   statusNotFound=1

   if [ ! -d "$nikalClientDB" ]; then
      # cache doesn't exist yet so nothing is installed
      return $statusNotFound
   fi

   # -e without -d will search the entire cache
   if [ -z "$clientInstallPath" ]; then
      clientFoundName=$(find $nikalClientDB -name $clientModuleUnversioned)
      if [ -z "$clientFoundName" ]; then
         return $statusNotFound
      else
         clientInstallPath=$(echo "$clientFoundName" | sed s#^${nikalClientDB}/##)
         clientInstallPath=$(echo "$clientInstallPath" | sed s#/${clientModuleUnversioned}##)

         # if clientModule was installed previously without -d
         if [ "$clientInstallPath" = "$clientModuleUnversioned" ]; then
            clientInstallPath=""
         fi
      fi
   fi

   if ! nikalGetConfiguredKernelModuleBasePath; then
      return 1
   fi

   if [ ! -L "$nikalClientDB/$clientInstallPath/$clientModuleUnversioned" ] ||
      [ ! -r "$moduleBasePath/$clientInstallPath/$clientModule" ]; then
      # not in the cache or not installed
      return $statusNotFound
   fi

   return $statusFound
}

#
# global variables
#
moduleOperation=""
clientInstallPath=""
clientModulePath=""
clientModuleUnversioned=""
clientModule=""
moduleOutputFile=""
ldSourceFile=""
# default namespace, NI products shouldn't need to change this
# namespace in KAL has been deprecated. This is left here, but unused
namespace=nNIPAL100
atomicInterfaceSymbol=nNIAPALS100_driverEntry
kalInterfaceSymbol=_driverEntry
clientKalSymbol=_ckalcbInitModule
# Boolean flags, 0 means on/true/yes/enabled, 1 means off/false/no/disabled
fastInstall=1
promptUser=1
forceRebuild=0
# CAR 525837 - DAQ driver versioning takes too long on first-boot when
# installing many drivers
# This workaround restores the original KAL behavior of building modules
# at module registration time.
buildDuringInstall=0

# Look up nikalDir and nikalClientDB
if ! nikalGetDirs; then
   error "Unable to locate a valid NI-KAL installation"
   exit 1
fi

#
# begin script execution
#

if ! parseArguments $*; then
   error "An error occurred while processing program arguments."
   exit 1
fi

if ! requirecommands id; then
   error "Some required tools are missing or were not found."
   exit 1
fi

# must be root except when running output
if ! isRoot && [ "$moduleOperation" != "output" ]; then
   error "Please run NI-KAL client kernel module installer as root"
   exit 1
fi

# Simple module registration and unregistration requires very little setup
# If that's all we're doing, then let's do it and be done
if [[ "${moduleOperation}" == "install" ]] && [[ "${buildDuringInstall}" = "1" ]]; then
   installKernelDriver
   exit $?
fi
if [[ "${moduleOperation}" == "remove" ]]; then
   removeKernelDriver
   exit $?
fi

# Set up the kernel driver versioning environment
nikalEnableKernelVersioning
if ! withVersioning configuredForCurrentKernel; then
   warning "Configured for kernel version $KERNELTARGET, which is not the currently running kernel"
fi

# Execute the requested operation
withVersioning ${moduleOperation}KernelDriver
ret=$?

nikalDisableKernelVersioning
exit $ret

