#!/bin/bash
# Copyright 2013,
# National Instruments Corporation.
# All rights reserved.

# Used to determine module dependency order
declare -A undefined
declare -A exported
declare -a modules
declare -A modules_inv

# Used to determine symbols that need to be renamed to fit within length limits
declare -i maxSymbolNameLength=55

isLinkBroken () {
	readlink -q $1 >/dev/null
}

removeBrokenLink () {
	rm -f "$1" || return $?
	#if directory containing link is empty, remove directory containing link
	rmdir -f "$(dirname $1)" >&/dev/null
	return 0
}

# printExportedSyms () - returns exported symbols for module
# $1 - module binary
printExportedSyms () {
	strings $1 | sed -ne \
		'/nNIExport_beginExportedSymbolsArray/ {
			n
			:a
			/nNIExport_endExportedSymbolsArray/q
			p
			n
			ba
		}'
}

# printImportedSyms () - returns imported (undefined) symbols for module
# $1 - module binary
printImportedSyms () {
	nm -u $1 | awk '{print $2}'
}

# isPCIDriver() - returns 0 if driver has a PCIIDTable
# $1 - module binary
isPCIDriver () {
   test `strings $1 | grep 'nNIKAL200_beginPCIIDTable' | wc -l` -gt 0
}

# printPCITable () - returns hw PCI IDs for module
# $1 - module binary
printPCITable () {
	strings $1 | sed -ne \
		'/nNIKAL200_beginPCIIDTable/ {
			n
			:a
			/nNIKAL200_endPCIIDTable/q
			p
			n
			ba
		}'
}

# isACPIDriver() - returns 0 if driver has a ACPIIDTable
# $1 - module binary
isACPIDriver () {
   test `strings $1 | grep 'nNIKAL1500_beginACPIIDTable' | wc -l` -gt 0
}

# printACPITable () - returns hw ACPI IDs for module
# $1 - module binary
printACPITable () {
	strings $1 | sed -ne \
		'/nNIKAL1500_beginACPIIDTable/ {
			n
			:a
			/nNIKAL1500_endACPIIDTable/q
			p
			n
			ba
		}'
}

# updateSymbolRenameFile () - updates symbols file with exported functions of given module
# $1 - symbols file (shortsyms.txt)
# $2 - module binary
updateSymbolRenameFile () {
	local symfilename="$1"
	local fullmodname="$2"
	modname="${fullmodname%%-unversioned.o}"
	modname="${modname##*/}"

	# Abort if module binary doesn't exist
	test -e ${fullmodname} || return 1

	# Abort if output file cannot be created/written
	touch ${symfilename} || return 1

	for symbol in $(printExportedSyms "${fullmodname}") $(printImportedSyms "${fullmodname}"); do
		if (( ${#symbol} > $maxSymbolNameLength )); then
			newsym="_$(md5sum <<< $symbol | cut -d ' ' -s -f 1)"
			printf "%s %s\n" "$symbol" "$newsym"
		fi
	done > ${symfilename}
	# This function always returns success if it reaches this point
	return 0
}

# tsortFunc () - sorts and displays all modules by given dependency array (topological sort),
# while modifying <modules_inv> associative array
# $1 - array with line-by-line dependencies
# the complexity of this function is O(n^2) reorderings and O(n) rescans of the input,
# but we are dealing with tens of dependencies, not thousands, so it's fine
tsortFunc () {
	local -i notDone=1
	while (( $notDone )); do
		(( notDone = 0 ))
		while read expSymModule module; do
			if [[ -n "$module" ]]; then
				# $module depends on $expSymModule
				if (( ${modules_inv[$module]} <=  ${modules_inv[$expSymModule]} )); then
					(( modules_inv[$module]= ${modules_inv[$expSymModule]} + 1 ))
					(( notDone = 1 ))
				fi
			fi
		done <<< "$1"
	done

	echo "$( for module in ${!modules_inv[@]}; do
		printf '%03d %s\n' ${modules_inv[$module]} $module
	done | sort | awk '{print $2}')"
}

# sortModules () - sorts all modules by dependency (topological sort)
# uses global variables <undefined> and <exported> passed from sortDependency ()
# 	to determine module dependencies
sortModules () {
	local dependencies=$(
		for module in "${!undefined[@]}"; do
			for symbol in ${undefined[${module}]}; do
				local expSymModule="${exported[${symbol}]}"
				if [[ -n $expSymModule ]]; then
					# $module depends on $expSymModule, print dependency
					printf '%s %s\n' "$expSymModule" "$module"
				fi
			done
		done | sort | uniq)

	tsortFunc "$dependencies"
}

# sortDependency () - sorts all modules in given directory by dependency
# $1 - directory containing all modules
sortDependency ()
{
	while IFS= read -r -d $'\0' l; do
		isLinkBroken "$l" || { removeBrokenLink "$l" && continue; }

		modules+=("$l")

		local module="$l"
		undefined[${module}]=`nm -u $module | awk '{print \$2}'`

		for expsym in $(printExportedSyms $module); do
			exported[${expsym}]=$module
		done
	done < <(find $1 -type l -print0)

	for i in ${!modules[@]}; do
		modules_inv["${modules[$i]}"]=$i
	done

	sortModules
	return $?
}

# printInterface () - generate wrapper from module
# $1 - module binary
printInterface ()
{
	local readonly moduleFile="$1"

	local exportedSyms="$(printExportedSyms $moduleFile)"
	local pciTable="$(printPCITable $moduleFile)"
	local acpiTable="$(printACPITable $moduleFile)"

	readonly -A interfaceTypeMap=(
	[nNIAPAL200_driverEntry]=atomicInterface
	[nNIPAL100_driverEntry]=kalInterface
	[nNIPAL100_ckalcbInitModule]=clientKalInterface)
	# determine client interface interfaceType
	interfaceType=
	while read line; do
		interfaceType=${interfaceTypeMap[$line]}
		[[ -z $interfaceType ]] || break
	done <<< "$exportedSyms"

	if  [[ -z $interfaceType ]]; then
		echo "Unknown Interface Type !!!" | tee -a /tmp/processmodulelog.txt
		return 1
	fi

	modname=$(basename ${moduleFile%-unversioned\.o})

	#
	# Now generate module-interface.c
	#

	cat << "EOF"

#undef __NO_VERSION__
#include <linux/version.h>

#ifdef nNIKAL160_kConfig
   #include <linux/config.h>
#endif
#include <linux/string.h>
#include <linux/errno.h>
#ifdef MODULE
   #if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
      #define MODVERSIONS
   #endif

   #ifdef MODVERSIONS
      #include <config/modversions.h>
   #endif

   #include <linux/module.h>
#endif

#include "nikal.h"

#ifdef MODULE_LICENSE
   #ifdef nNIKAL100_kCopyrightString
      MODULE_LICENSE(nNIKAL100_kCopyrightString);
   #endif
#endif

#ifdef MODULE_AUTHOR
   #ifdef nNIKAL100_kAuthorString
      MODULE_AUTHOR(nNIKAL100_kAuthorString);
   #endif
#endif

#ifdef MODULE_DESCRIPTION
   #ifdef nNIKAL100_kDescriptionString
      MODULE_DESCRIPTION(nNIKAL100_kDescriptionString);
   #endif
#endif

EOF
	if [[ $interfaceType = "kalInterface" ]]; then
		cat << "EOF"
static struct nNIKAL100_tDriver kalDriver;
nNIKAL100_cc int nNIPAL100_driverEntry(struct nNIKAL100_tDriver *driver);

int init_module(void)
{
#ifdef __i386__
   /** We don't support machine configuration with physical memory located
    *  above the 4GB physical address boundary on 32bit systems.
    *  This typically occurs for systems that have more than 3GB memory.  The
    *  BIOS needs to reserve some physical address space for device mapping,
    *  so it puts any additional memory beyond 3GB above the 4GB physical
    *  address space.
    *  Below, we check if addressableMemory overflows a 32 bit value.
    *  On 32bit systems this occurs when there is more than 4GB of addressable memory.
    **/

   // PAL is the only one that use this. Fail 32bit PAL load if amount of addressable memory is > 4G, since PAL can't handle it.
   if (nNIKAL200_isAddressableMemOver4G())
   {
      nNIKAL100_printToDebugger("[nipal] More than 4GB of addressable memory detected.\n");
      nNIKAL100_printToDebugger("[nipal] This configuration is not supported.  Check the release notes for more information.\n");
      return -ENOSYS;
   }
#endif

   /* We need to initialize the client first before registering our driver to the
    * kernel. This is to guarantee that we're ready to receive connection from
    * user mode right after we register to the kernel. */
   if (nNIPAL100_driverEntry(&kalDriver))
      return -ENODEV;
   kalDriver.module = THIS_MODULE;

   /* on fail registerDriver will call unregisterDriver which will
    * subsequently call driver->unload for us */
   return nNIKAL100_registerDriver(&kalDriver);
}

void cleanup_module(void)
{
   /* unregisterDriver also calls driver->unlod for us */
   nNIKAL100_unregisterDriver(&kalDriver);
}

EOF
	elif [[ $interfaceType = "clientKalInterface" ]]; then
		cat << "EOF"
nNIKAL100_cc int nNIPAL100_ckalcbInitModule(void* modulePtr, const char* name);
nNIKAL100_cc void nNIPAL100_ckalcbCleanupModule(void* modulePtr);

int init_module(void)
{
   return nNIPAL100_ckalcbInitModule(THIS_MODULE,(THIS_MODULE)->name);
}

void cleanup_module(void)
{
   nNIPAL100_ckalcbCleanupModule(THIS_MODULE);
}

EOF
	elif [[ $interfaceType = "atomicInterface" ]]; then
		# Conditionally include the correct headers for the type of driver
		isPCIDriver $moduleFile && echo '#include <linux/pci.h>'
		if isACPIDriver $moduleFile; then
			echo '#include <acpi/acpi.h>'
			echo '#include <acpi/acpi_bus.h>'
			echo '#include <acpi/acpi_drivers.h>'
		fi

		# Generate driver entrypoints
		cat << "EOF"
#include <linux/mod_devicetable.h>

nNIKAL100_cc int nNIAPAL200_driverEntry(struct nNIKAL100_tDriver *driver);

static struct nNIKAL100_tDriver atomicDriver;
static void* nativeDriverRegistration;

/* native driver registration functions. These functions are auto-generated */
void *registerNativeDriver(nNIKAL100_tDriver*);
void unregisterNativeDriver(void*);

int init_module(void)
{
   atomicDriver.module = THIS_MODULE;

   if (nNIAPAL200_driverEntry(&atomicDriver))
      return -ENODEV;

   nativeDriverRegistration = registerNativeDriver(&atomicDriver);

   if (nativeDriverRegistration == NULL)
   {
      if (atomicDriver.unload)
         atomicDriver.unload(&atomicDriver);
      return -ENODEV;
   }

   return 0;
}

void cleanup_module(void)
{
   unregisterNativeDriver(nativeDriverRegistration);

   if (atomicDriver.unload)
      atomicDriver.unload(&atomicDriver);
}

EOF
		if isPCIDriver $moduleFile; then
			#
			# generate hw ID table and native driver registration
			#
			cat << "EOF"
static struct pci_device_id nipciidtable[] = {
EOF
			if  [[ -n "$pciTable" ]]; then
				while read line; do
					printf "   {"
					i=0
					for id in $line; do
						if [[ $id = "none" ]]; then
							if (( i == 4 || i == 5 )); then
								# these are pci class spec. use '0' instead of None
								printf "0, "
							else
								printf "PCI_ANY_ID, "
							fi
						else
							printf "%s, " "$id"
						fi
						(( i++ ))
					done
					printf "(unsigned long)&atomicDriver},\n"
				done <<< "$pciTable"
			fi
			cat << EOF

   { },
};
MODULE_DEVICE_TABLE(pci, nipciidtable);

static struct pci_driver niPciDriver = {
   .name = "$modname",
   .id_table = nipciidtable,
};

void *registerNativeDriver(nNIKAL100_tDriver* ignore)
{
   return nNIKAL200_registerPCIDriver(&niPciDriver);
}
void unregisterNativeDriver(void* data)
{
   nNIKAL200_unregisterPCIDriver(data);
}

EOF
		elif isACPIDriver $moduleFile; then
			#
			# generate hw ID table and native driver registration
			#
			cat << "EOF"
static const struct acpi_device_id niacpiidtable[] = {
EOF
			if  [[ -n "$acpiTable" ]]; then
				while read line; do
					printf "   {"
					i=0
					for id in $line; do
						printf "\"%s\", " "$id"
						(( i++ ))
					done
					printf "(unsigned long)&atomicDriver},\n"
				done <<< "$acpiTable"
			fi
			cat << EOF
   { },
};
MODULE_DEVICE_TABLE(acpi, niacpiidtable);

static struct acpi_driver niAcpiDriver = {
   .name = "$modname",
   .class= "National Instruments",
   .ids  = niacpiidtable,
};

static int acpiProbe(void *dev)
{
   return nNIKAL1500_helperACPIProbe(dev, &atomicDriver);
}

void *registerNativeDriver(nNIKAL100_tDriver* ignore)
{
   return nNIKAL1500_registerACPIDriver(&niAcpiDriver, &acpiProbe);
}

void unregisterNativeDriver(void* data)
{
   nNIKAL1500_unregisterACPIDriver(data);
}

EOF
		else
			# no ID Table
			cat << "EOF"
/* native driver registration is no-op */
void *registerNativeDriver(struct nNIKAL100_tDriver* ignore)
{
   static int some_non_null_storage_ignoreme;
   return &some_non_null_storage_ignoreme;
}
void unregisterNativeDriver(void* ignore) { }

EOF

		fi # isPCIDriver/isACPIDriver
	fi # [[ $interfaceType = "atomicInterface" ]]

	#
	# generate symbol exports
	#
	cat << "EOF"
#ifdef EXPORT_SYMBOL_NOVERS
#define nNIKAL100_mExportSymbol EXPORT_SYMBOL_NOVERS
#else
#define nNIKAL100_mExportSymbol EXPORT_SYMBOL
#endif
EOF

	#
	# generate export sym section
	#
	i=0
	for sym in $exportedSyms; do
		if ! [[ $sym =~ _(driverEntry|ckalcbInitModule|ckalcbCleanupModule)$ ]]; then
			if (( ${#sym} > $maxSymbolNameLength )); then
				newsym="_$(md5sum <<< $sym | cut -d ' ' -s -f 1)"
				printf '/* %s=%s */\n' "$sym" "$newsym"
			else
				newsym=$sym
			fi
			printf 'extern int %s;\nnNIKAL100_mExportSymbol(%s);\n' "$newsym" "$newsym"
		fi
	done
}

parseOptHelp()
{
	cat << "EOF"
Usage: processmodule.sh --cmd_sortdependency --kaldbdir=DIRECTORY
       processmodule.sh --cmd_updatesystemrename --symfile=FILE
       processmodule.sh --cmd_geninterface --module=MODULE

Helper script for kernel module installation for National Instruments

Options:
  -h, --help            show this help message and exit
  --cmd_sortdependency  print out the modules sorted in dependency, for all
                        modules in kaldbdir
  --cmd_updatesymrename
                        update the file that contains the list of symbols that
                        needs renaming
  --cmd_geninterface    generate the interface file and print to stdout
  --kaldbdir=KALDBDIR   ni-kal client db directory
  --symfile=SYMFILE     the symbol file to use
  --module=MODULE       the module path
EOF
}

# parse command options and run necessary use case
# $1 - all options passed to this script
parseOptAndRun()
{
	local -i v_sortdependency=0
	local -i v_updatesymrename=0
	local -i v_geninterface=0
	local v_kaldbdir=
	local v_symfile=
	local v_module=

	while test $# -gt 0; do
		case "$1" in
			-h | --help)
				parseOptHelp
				return 0
				;;
			--cmd_sortdependency)
				v_sortdependency=1
				;;
			--cmd_updatesymrename)
				v_updatesymrename=1
				;;
			--cmd_geninterface)
				v_geninterface=1
				;;
			--kaldbdir=*)
				v_kaldbdir=${1#*=}
				;;
			--symfile=*)
				v_symfile=${1#*=}
				;;
			--module=*)
				v_module=${1#*=}
				;;
		esac
		shift
	done

	if (( "$v_sortdependency" == 0 && "$v_updatesymrename" == 0 && "$v_geninterface" == 0 )); then
		echo -e "\nYou must specify only one of the commands\n"
		parseOptHelp
		return 1
	fi

	if (( "$v_sortdependency" == 1 )); then
		if [[ -n "$v_kaldbdir" ]]; then
			sortDependency $v_kaldbdir
			return $?
		else
			echo -e "\nNo value given to --kaldbdir\n"
		fi
	elif (( "$v_updatesymrename" == 1 )); then
		if [[ -n "$v_symfile" && -n "$v_module" ]]; then
			updateSymbolRenameFile $v_symfile $v_module
			return $?
		else
			echo -e "\nNo value given for either --symfile or --module\n"
		fi
	elif (( $v_geninterface == 1 )); then
		if [[ -n "$v_module" ]]; then
			printInterface $v_module
			return $?
		else
			echo -e "\nNo value given to --module\n"
		fi
	fi

	parseOptHelp | tee -a /tmp/processmodulelog.txt
	return 1
}

echo "$0 $@" >> /tmp/processmodulelog.txt

# parse command options and run necessary use case
if parseOptAndRun $@; then
   rm -f /tmp/processmodulelog.txt > /dev/null 2>&1
else
   exit $?
fi

exit 0

