#!/bin/bash
#
# installerUtility.sh
# This is the installation utility script for NI-KAL for Linux.
# This script is used by the nikal INSTALL script and can be sourced
# by other installers to call the functions it provides.
#
#   Copyright 2005,
#  National Instruments Corporation.
#  All Rights reserved.
#
#  originated:  4 February 2005
#

nikalVersion=15.0.1f0

#
# Logging functions
#

# Logs a message to the system log (if "logger" is available) and/or stderr
# with stdout as the fallback.
# Inputs:
#     LOG_MSG_TYPE   - the "logger" log facility and level, also called
#                      "priority" - see "man logger" for more detail
#     LOG_MSG_STDERR - whether the message should be printed to stderr
#                      if "logger is available, the message will be logged
#                      and written to stderr, otherwise the message will just
#                      be written to stderr
#     LOG_MSG_TAG    - a tag to prefix the message with, only used with logger
# Outputs:
if command -v logger > /dev/null 2>&1; then
   function log_msg() {
      logger ${LOG_MSG_TYPE+-p "$LOG_MSG_TYPE"} ${LOG_MSG_TAG+-t "$LOG_MSG_TAG"} ${LOG_MSG_STDERR+-s} "$@"
   }
else
   function log_msg() {
      if [[ -n "${LOG_MSG_STDERR}" ]]; then
         echo "$@" >&2
      else
         echo "$@"
      fi
   }
fi

# Logs the contents of a file to the system log (if "logger" is available) and/or stderr
# with stdout as the fallback.
# Inputs:
#     LOG_MSG_TYPE   - the "logger" log facility and level, also called
#                      "priority" - see "man logger" for more detail
#     LOG_MSG_STDERR - whether the message should be printed to stderr
#                      if "logger is available, the message will be logged
#                      and written to stderr, otherwise the message will just
#                      be written to stderr
#     LOG_MSG_TAG    - a tag to prefix the message with, only used with logger
# Outputs:
if command -v logger > /dev/null 2>&1; then
   function log_file() {
      logger ${LOG_MSG_TYPE+-p "$LOG_MSG_TYPE"} ${LOG_MSG_TAG+-t "$LOG_MSG_TAG"} ${LOG_MSG_STDERR+-s} < "$@"
   }
else
   function log_file() {
      if [[ -n "${LOG_MSG_STDERR}" ]]; then
         cat "$@" >&2
      else
         cat "$@"
      fi
   }
fi

# Log an error message to the system log (if available) and to stderr
# Inputs:
#     LOG_MSG_TAG    - a tag to prefix the message with, only used with logger
#     $@             - the error message, an "ERROR: " prefix will be
#                      added automatically
# Outputs:
function error() {
   LOG_MSG_TYPE="user.err" LOG_MSG_STDERR="1" log_msg "ERROR:" "$@"
}

# Log the contents of a file as an error message
# Inputs:
#     LOG_MSG_TAG    - a tag to prefix the message with, only used with logger
#     $@             - the error message, an "ERROR: " prefix will be
#                      added automatically
# Outputs:
function errorfile() {
   LOG_MSG_TYPE="user.err" LOG_MSG_STDERR="1" log_file "$@"
}

# Log a warning message to the system log (if available) and to stderr
# Inputs:
#     LOG_MSG_TAG    - a tag to prefix the message with, only used with logger
#     $@             - the error message, a "WARNING: " prefix will be
#                      added automatically
# Outputs:
function warning() {
   LOG_MSG_TYPE="user.warning" LOG_MSG_STDERR="1" log_msg "WARNING:" "$@"
}

# Log an informational message to stdout
# Inputs:
#     $@             - the info message
# Outputs:
function info() {
   #LOG_TYPE_TYPE="user.info" log_msg "$@"
   echo "$@"
}

#
# Command utility functions
#

# Look up the path to the specified utility
# Inputs:
#     $1             - the name of the utility to find
# Outputs:
#     FINDUTIL       - the path to the found utility, or if a built-in,
#                      the name of the utility, or if not found,
#                      an empty string
# Returns: 0=success, nonzero=failure
function findutil() {
   # Validate inputs
   if [ -z "$1" ]; then
      return 1
   fi
   FINDUTIL=""
   # Not all commands are executables - some are bash builtins
   if command -v $1 > /dev/null 2>&1; then
      if which $1 >/dev/null 2>&1; then
         # Use which to locate the executable
         FINDUTIL=$(which $1 2>/dev/null)
      else
         # Bash builtin
         FINDUTIL=$1
      fi
   else
      local notfound_msg
      printf -v notfound_msg "%-10s Not found in current path" "$1"
      info "$notfound_msg"
      return 1
   fi
   return 0
}

# Test that the specified commands exist
# Inputs:
#     $@             - the name(s) of the command(s) to test for
# Outputs:
#     (UTILITY)      - the all-caps name of the utility is set to the
#                      fully-qualified path or name of the command if it exists
# Returns: 0=success, nonzero=failure
function requirecommands() {
   local ret=0
   for cmd in $*; do
      if findutil $cmd; then
         cmdvar=${cmd^^}
         eval "$cmdvar=$FINDUTIL"
      else
         ret=$(($ret + $?))
      fi
   done
   return $ret
}

# test for the existence of various common, unix utilities required by NI-KAL
# Inputs:
#     NO_INSTALL_UTIL - do not require the "install" command (not available in
#                       some embedded environments)
#     NO_COMPILE_UTIL - do not require compiler and build tools
# Outputs:
# Returns: 0=success, nonzero=failure
# Unless NO_COMPILE_UTIL is set, requires a versioning-capable environment
function verifyRequiredTools() {
   #
   # Verify tools used during install
   #
   local requiredcommands=(cat cut sed grep awk strings wc
                           mv cp rm mkdir rmdir chmod mknod)
   # The "install" utility is missing in many embedded linuxes
   if [ "$NO_INSTALL_UTIL" != "1" ]; then
      requiredcommands+=(install)
   fi

   # Misc utils
   requiredcommands+=(expr id uname)
   # Some users of this script don't have access to compile tools during install
   if [ "$NO_COMPILE_UTIL" != "1" ]; then
      requiredcommands+=(make gcc objcopy)
   fi

   if ! requirecommands ${requiredcommands[@]}; then
      error "Some required tools are missing or were not found."
      return 1
   fi
}

# Output tools found
# Print the names and paths (in a form suitable for inclusion by bash or Make)
# of various required tools
# Inputs:
#     NO_INSTALL_UTIL - do not require the "install" command (not available in
#                       some embedded environments)
#     NO_COMPILE_UTIL - do not require compiler and build tools
# Outputs:
#     stdout          - assignment of variables of commands with the paths
#                       of those commands
# Returns: 0=success, nonzero=failure
# Unless NO_COMPILE_UTIL is set, requires a versioning-capable environment
function dumpTools() {
   if ! verifyRequiredTools > /dev/null 2>&1; then
      return $?
   fi

   cat <<TOOL_DEFINITIONS
      AWK=$AWK
      CAT=$CAT
      CC=$GCC
      MAKE=$MAKE
      GMAKE=$MAKE
      GCC=$GCC
      CHMOD=$CHMOD
      CP=$CP
      CUT=$CUT
      EXPR=$EXPR
      GREP=$GREP
      ID=$ID
      INSTALL=$INSTALL
      MKDIR=$MKDIR
      MKNOD=$MKNOD
      MV=$MV
      OBJCOPY=$OBJCOPY
      RM=$RM
      RMDIR=$RMDIR
      SED=$SED
      STRINGS=$STRINGS
      UNAME=$UNAME
      WC=$WC
TOOL_DEFINITIONS
}

# versionCompare() - compares two RPM-style version numbers
# and returns 255, 0, 1, or 2 to indicate if the first verison is
# less than (255), equal to (0), or greater than (1) the second,
# or an error occurred (2). Version numbers can be specified with or without
# release suffixes.  Alpha, beta, etc. suffixes should always specified be in
# the release portion.
#
# Examples:
#  versionCompare 1.0  1.0.1
#  versionCompare 1.0-a1 1.0-a12
#  versionCompare 1.0.1-b1 1.0-1
#  versionCompare 2.2.3.4-99  2.2.4
#  versionCompare 8.0-b129 8.0-rc0
#  versionCompare 8.0-rc0 8.0-1
function versionCompare() {
   if ! requirecommands sed sort; then
      return 1
   fi

   function canonicalizeVersion() {
      canonicalVersion=$(echo "$1" |
         sed -e 's,\([0-9]\)\([A-Za-z]\),\1.000.\2,g' \
             -e 's,\([A-Za-z]\)\([0-9]\),\1.\2,g' \
             -e 's,^\([A-Za-z]\),---.\1,' \
             -e 's,\b\([0-9]\)\b,00\1,g' \
             -e 's,\b\([0-9][0-9]\)\b,0\1,g')
   }

   local arg1 arg2 nResult firstVersion secondVersion resultVersion
   if [ "$1" = "$2" ]; then
      return 0;
   fi
   arg1=$(echo "$1" | sed 's,-.*$,,')
   arg2=$(echo "$2" | sed 's,-.*$,,')
   if [ "$1" != "$arg1" -o "$2" != "$arg2" ]; then
      versionCompare "$arg1" "$arg2"
      nResult=$?
      if [ $nResult -ne 0 ]; then
         return $nResult
      fi
   fi
   canonicalizeVersion $(echo "$1" | sed 's,^.*-,,')
   firstVersion="$canonicalVersion"
   canonicalizeVersion $(echo "$2" | sed 's,^.*-,,')
   secondVersion="$canonicalVersion"
   resultVersion=$( (echo "$firstVersion"; echo "$secondVersion") |
                     LC_ALL=C sort -t. -k 1 -k 2 -k 3 -k 4 -k 5 -k 6 -k 7 -k 8 -k 9 -k 10 |
                     head -1)
   if [ "$firstVersion" = "$resultVersion" ]; then
      return 255
   else
      return 1
   fi
}

# Determine if the running user is root
# Inputs:
# Outputs:
# Returns: 0=root, nonzero=nonroot
function isRoot() {
   test "$EUID" -eq 0
}

# Displays supported linux distribution information
# Inputs:
# Outputs:
# Returns: 0=success
function niDistributionCheck() {
   info "National Instruments products support the following Linux distributions:"
   info "   NI Linux-RT"
   info "   Red Hat Enterprise Linux Desktop + Workstation"
   info "   CentOS Linux"
   info "   openSUSE"
   info "Refer to README.txt for the latest information at the time of release."
   info "Refer to www.ni.com/linux for the most recent information about Linux."
   info "support at National Instruments."
   return 0
}

# Default implementation of withVersioning for normal linux
# environments where kernel sources and compilers and things
# are all exactly where they ought to be
# Inputs:
# Outputs:
# Returns: nonzero=failure
# Requires a versioning-capable environment
function withVersioning() {
   if type -t versioning_call > /dev/null 2>&1; then
      # Older Linux-RT (2014) versions don't set sysroot, so well
      # Set it ourselves, and let versioning_call override it if it needs to
      CFLAGS=--sysroot="${TOOLS_MNTPNT}" versioning_call "$@"
   else
      "$@"
   fi
}

# Set up the linux-rt module versioning environment
# Inputs:
# Outputs:
# Returns: 0=success, nonzero=failure
function nikalEnableKernelVersioning() {
   local versioning_script="/usr/local/natinst/tools/versioning_utils.sh"
   if [ -r "${versioning_script}" ]; then
      # Source the utility script to get things setup for versioning
      source "${versioning_script}"

      # Set the stage for doing the actual versioning
      setup_versioning_env > /dev/null 2>&1

      # Set the tools-supplied kernel version to build against
      if [ -z "$KERNELTARGET" ]; then
         export KERNELTARGET=$(kernel_version)
         nikalGetRequestedKernelVersion
      fi
   fi
}

# Tears down the linux-rt module versioning environment
# Inputs:
# Outputs:
# Returns: 0=success, nonzero=failure
function nikalDisableKernelVersioning() {
   local versioning_script="/usr/local/natinst/tools/versioning_utils.sh"
   if [ -r "${versioning_script}" ]; then
      cleanup_versioning_env > /dev/null 2>&1
   fi
}

# Fills out information about the requested kernel version
# Inputs:
#     KERNELTARGET    - (optional - default is $(uname -r))
# Outputs:
#     kernelVersion - the version (name) of the kernel to be used
#     kernelMajor   - the major version of the kernel to be used
#     kernelMinor   - the minor version of the kernel to be used
#     kernelPatchLevel - the update (patch-level) version of the kernel to be used
# Returns: 0=success, nonzero=failure
function nikalGetRequestedKernelVersion() {
   kernelVersion=
   kernelMajor=
   kernelMinor=
   kernelPatchLevel=

   if ! requirecommands cut; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   if [ -n "$KERNELTARGET" ]; then
      kernelVersion=$KERNELTARGET
   else
      kernelVersion=$(uname -r)
   fi
   kernelMajor=$(echo $kernelVersion | cut -d "-" -f 1 | cut -d "." -f 1)
   kernelMinor=$(echo $kernelVersion | cut -d "-" -f 1 | cut -d "." -f 2)
   kernelPatchLevel=$(echo $kernelVersion | cut -d "-" -f 1 | cut -d "." -f 3)
   return 0
}

# Fills out information about the module directory where built modules
# would be installed.
# Inputs:
#     INSTALL_MOD_PATH- (optional - default is "")
#     KERNELTARGET    - (optional - default is $(uname -r))
# Outputs:
#     kernelmodDir  - the path to where built kernel modules are installed
# Returns: 0=success, nonzero=failure
# Requires a versioning-capable environment
function nikalGetRequestedKernelModulesDir() {
   kernelmodDir=

   #   Set INSTALL_MOD_PATH to change the base path for the modules to install
   #   modules under.  This is used when building modules for a kernel not installed
   #   on the system (cross-compiling, building against a git kernel, etc).
   # Test if kernelVersion is already set, otherwise, look it up
   if [ -z "$kernelVersion" ] && ! nikalGetRequestedKernelVersion; then
      return $?
   fi
   kernelmodDir=$INSTALL_MOD_PATH/lib/modules/$kernelVersion
   if [ -d "$kernelmodDir" ]; then
      return 0
   else
      error "$kernelmodDir directory not found."
      return 1
   fi
}

# Fills out information about where the configured kernel sources are located
# Inputs:
#     INSTALL_MOD_PATH- (optional - default is "")
#     KERNELTARGET    - (optional - default is $(uname -r))
#     KERNELHEADERS   - (optional - default is $INSTALL_MOD_PATH/lib/modules/$KERNELTARGET/...)
# Outputs:
#     headersDir      - the path to the configured kernel sources
# Returns: 0=success, nonzero=failure
# Requires a versioning-capable environment
function nikalGetRequestedKernelSourcesDir() {
   headersDir=

   if [ -n "$KERNELHEADERS" ]; then
      headersDir=$KERNELHEADERS
      return 0
   fi

   # Test if kernelmodDir is already set, otherwise, look it up
   if [ -z "$kernelmodDir" ] && ! nikalGetRequestedKernelModulesDir; then
      return $?
   fi

   # Test if kernelVersion is already set, otherwise, look it up
   if [ -z "$kernelVersion" ] && ! nikalGetRequestedKernelVersion; then
      return $?
   fi

   if nikalKernelSourcesAreConfigured "$kernelmodDir/build"; then
      headersDir="$kernelmodDir/build"
      return 0
   fi

   if nikalKernelSourcesAreConfigured "$kernelmodDir/source"; then
      headersDir="$kernelmodDir/source"
      return 0
   fi

   if nikalKernelSourcesAreConfigured "/usr/src/linux-$kernelVersion"; then
      headersDir="/usr/src/linux-$kernelVersion"
      return 0
   fi

   if nikalKernelSourcesAreConfigured "/usr/src/linux-$kernelMajor.$kernelMinor"; then
      headersDir="/usr/src/linux-$kernelMajor.$kernelMinor"
      return 0
   fi

   if nikalKernelSourcesAreConfigured "/usr/src/kernels/$kernelVersion"; then
      headersDir="/usr/src/kernels/$kernelVersion"
      return 0
   fi

   if nikalKernelSourcesAreConfigured "/usr/src/linux"; then
      headersDir="/usr/src/linux"
      return 0
   fi
   return 1
}

# Determines whether the specified kernel sources path is fully configured
# Inputs:
#     $1              - the path to the candidate kernel sources
# Outputs:
# Returns: 0=fully-configured, nonzero=not configured
# Requires a versioning-capable environment
function nikalKernelSourcesAreConfigured() {
   if ! requirecommands uname egrep; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   local kernelDir=$1
   local asmDir=
   if [ -z "$kernelDir" ] || [ ! -d "$kernelDir" ]; then
      return 1
   fi

   if [ ! -f $kernelDir/Module.symvers ]; then
      return 1
   fi

   if [ ! -f $kernelDir/.config ]; then
      return 1
   fi

   if [ ! -f "$kernelDir/include/linux/kernel.h" ]; then
      return 1
   fi

   if [ -d $kernelDir/include/asm ]; then
      asmDir=$kernelDir/include/asm
   elif [ -d $kernelDir/include/generated ]; then
      asmDir=$kernelDir/include/generated
   else
      return 1
   fi

   if uname -m | egrep "i.86" > /dev/null 2>&1; then
      if [ ! -f $asmDir/asm_offsets.h ] && [ ! -f $asmDir/asm-offsets.h ]; then
         return 1
      fi
   fi
   return 0
}

# Fills out information about where the configured kernel sources are located
# Inputs:
#     INSTALL_MOD_PATH- (optional - default is "")
#     KERNELTARGET    - (optional - default is $(uname -r))
#     KERNELHEADERS   - (optional - default is $INSTALL_MOD_PATH/lib/modules/$KERNELTARGET/...)
# Outputs:
#     utsRelease      - the configured kernel's version/name as specified in source
# Returns: 0=success, nonzero=failure
# Requires a versioning-capable environment
function nikalGetRequestedKernelUTSRelease() {
   utsRelease=
   # Test if headersDir is already set, otherwise, look it up
   if [ -z "$headersDir" ] && ! nikalGetRequestedKernelSourcesDir; then
      return $?
   fi

   if ! requirecommands grep; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   local versionFiles="$headersDir/include/linux/utsrelease.h
                       $headersDir/include/generated/linux/utsrelease.h
                       $headersDir/include/generated/utsrelease.h
                       $headersDir/include/linux/version.h
                       $headersDir/include/generated/uapi/linux/version.h"
   for versionFile in $versionFiles; do
      if [ -r "$versionFile" ]; then
         local tmp_utsRelease=$(grep -o "UTS_RELEASE.*" $versionFile | sed 's/.*"\(.*\)"[^"]*$/\1/')
         if [ -n "$tmp_utsRelease" ]; then
            utsRelease="$tmp_utsRelease"
            return 0
         fi
      fi
   done
   return 1
}

# Check the system for pre-requisites for installation.
# Inputs:
#     INSTALL_MOD_PATH- (optional - default is "")
#     KERNELTARGET    - (optional - default is $(uname -r))
#     KERNELHEADERS   - (optional - default is $INSTALL_MOD_PATH/lib/modules/$KERNELTARGET/...)
# Outputs:
# Returns: 0=success, nonzero=failure
# Requires a versioning-capable environment
function nikalInstallationCheck() {
   # check for gcc since we need that to successfully install
   if ! requirecommands gcc >/dev/null 2>&1; then
      error "gcc does not appear to be installed!"
      error "Installation of gcc is required to continue this installation."
      return 1
   fi

   if ! requirecommands grep; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   # Test if kernelVersion is already set, otherwise, look it up
   if [ -z "$kernelVersion" ] && ! nikalGetRequestedKernelVersion; then
      return $?
   fi

   # Test if kernelmodDir is already set, otherwise, look it up
   if [ -z "$kernelmodDir" ] && ! nikalGetRequestedKernelModulesDir; then
      return $?
   fi

   # Test if headersDir is already set, otherwise, look it up
   if [ -z "$headersDir" ] && ! nikalGetRequestedKernelSourcesDir; then
      local ret=$?
      error "Kernel source does not appear to be installed for the $kernelVersion kernel."
      error "Installation of the kernel-source package for kernel $kernelVersion is"
      error "required to continue this installation."
      return $ret
   fi

   if ! nikalKernelSourcesAreConfigured "$headersDir"; then
      local ret=$?
      error "Kernel source in $headersDir does not appear to be"
      error "configured for the $kernelVersion kernel."
      error "Configuration of kernel source is required to continue installation."
      error "Refer to the README file for the product you are installing for information"
      error "about configuring your kernel source."
      return $ret
   fi

   # check that the headers dir that we found is for the correct version of
   # the kernel
   # Test if utsRelease is already set, otherwise, look it up
   if [ -z "$utsRelease" ] && ! nikalGetRequestedKernelUTSRelease; then
      local ret=$?
      error "Unable to identify kernel version string for the $kernelVersion kernel."
      return $ret
   fi

   if [ -z "$utsRelease" ] || [ "$(echo \"$utsRelease\" | grep -c \"$kernelVersion\")" -eq "0" ]; then
      error "Kernel source in $headersDir does not appear to be"
      error "for the $kernelVersion kernel."
      error "Ensure that kernel source for the $kernelVersion kernel is installed"
      error "and configured.  Refer to the README file for the product you are"
      error "installing for information about configuring your kernel source."
      return 1
   fi

   return 0
}

# Look up the path to where NI-KAL is installed
# Inputs:
# Outputs:
#     nikalDir        - the path where NI-KAL is installed
#     nikalClientDB   - the path to where NI-KAL client modules are registered
# Returns: 0=success, nonzero=failure
function nikalGetDirs() {
   nikalDir=
   nikalClientDB=
   if ! requirecommands cat; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   if [ ! -f /etc/natinst/nikal/nikal.dir ]; then
      error "Could not find /etc/natinst/nikal.dir"
      return 1
   fi
   nikalDir=$(cat /etc/natinst/nikal/nikal.dir)
   if [ -n "$nikalDir" ]; then
      nikalClientDB=$nikalDir/etc/clientkdb
   elif [ -d "./clientkdb" ]; then
      nikalClientDB=clientkdb
   else
      error "Could not find the client module registration directory"
      return 1
   fi
   return 0
}

# Look up the path to the targeted kernel's NI-KAL configuration file
# Inputs:
# Outputs:
#     nikalMakefile   - the path to NI-KAL's build configuration include file
#                       will be empty if KAL hasn't been configured yet for that kernel
# Returns: 0=success, nonzero=failure
function nikalGetConfigureSettings() {
   nikalMakefile=
   nikalGetModuleBuildDir "nikal"
   if [ -n "$moduleBuildDir" ] && [ -r "$moduleBuildDir/Makefile.in" ]; then
      nikalMakefile=$moduleBuildDir/Makefile.in
   else
      error "Could not locate the current NI-KAL configuration for kernel version $kernelVersion"
      return 1
   fi
   return 0
}

# Look up the path to where NI-KAL client modules will be installed
# Inputs:
# Outputs:
#     moduleBasePath  - the path where built kernel modules are/will be installed
# Returns: 0=success, nonzero=failure
function nikalGetConfiguredKernelModuleBasePath() {
   moduleBasePath=
   if ! nikalGetConfigureSettings; then
      return $?
   fi

   if [ -r $nikalMakefile ]; then
      moduleBasePath=$(source $nikalMakefile && echo $MODPATH)
   else
      error "Error reading current NI-KAL configuration for kernel version $kernelVersion"
      return 1
   fi
   return 0
}

# Look up the path to the configured kernel sources KAL was configured for
# Inputs:
# Outputs:
#     kernelDir       - the path to the configured kernel sources used for build
# Returns: 0=success, nonzero=failure
function nikalGetConfiguredKernelDir() {
   kernelDir=
   if ! nikalGetConfigureSettings; then
      return $?
   fi

   if [ -r $nikalMakefile ]; then
      kernelDir=$(source $nikalMakefile && echo $KERNELDIRECTORY)
   else
      error "Error reading current NI-KAL configuration for kernel version $kernelVersion"
      return 1
   fi
   return 0
}

# Look up the name of the client module grouping directory
# Inputs:
#     $1              - the name of the kernel module
#     nikalClientDB   - the path to KAL's client module DB
# Outputs:
#     moduleGroupDir  - the grouping folder for the specified module
# Returns: 0=success, nonzero=failure
function nikalGetModuleGroupDir() {
   local moduleName=$1
   moduleGroupDir=

   nikalGetDirs

   if [ -d "$nikalClientDB" ]; then
      for moduleUnversioned in $(find $nikalClientDB/ -name "${moduleName}-unversioned.o"); do
         local clientModule=${moduleUnversioned#$nikalClientDB/}
         moduleGroupDir=${clientModule%\/*}
      done
   else
      error "Could not find the client module registration directory"
      return 1
   fi

   test -n "$moduleGroupDir"
   return $?
}

# Creates the kernel version-specific build directory for the specified module
# Inputs:
#     $1              - kernel module name
#     $2              - the path to the KAL client module -unversioned.o
#                       (unnecessary if kernel module name is "nikal")
#     kernelVersion   - the name of the configured kernel sources used for build
#     nikalDir        - the path where NI-KAL is installed
# Outputs:
#     echos the path to stderr
# Returns: 0=success, nonzero=failure
function nikalCreateModuleBuildDir() {
   local moduleName=$1
   local moduleUnversioned=$2

   # Test if kernelVersion is already set, otherwise, look it up
   if [ -z "$kernelVersion" ] && ! nikalGetRequestedKernelVersion; then
      return $?
   fi

   function installFile() {
      local filename=$1
      local sourcedir=$2
      local targetdir=$3
      if [[ ! -r "${targetdir}/${filename}" ]]; then
         cp -f "${sourcedir}/${filename}" "${targetdir}/"
      fi
   }

   nikalGetModuleBuildDir "$moduleName"
   mkdir -p "${moduleBuildDir}"
   if [[ "${moduleName}" = "nikal" ]]; then
      installFile configure   ${nikalDir}/src/nikal ${moduleBuildDir}
      installFile Makefile    ${nikalDir}/src/nikal ${moduleBuildDir}
      installFile nikal.h     ${nikalDir}/src/nikal ${moduleBuildDir}
      installFile nikal.c     ${nikalDir}/src/nikal ${moduleBuildDir}
      installFile nidevnode.c ${nikalDir}/src/nikal ${moduleBuildDir}
   else
      if [[ -z "$moduleUnversioned" ]]; then
         error "Missing required parameter to nikalCreateModuleBuildDir: path to client module blob."
         return 1
      fi
      # The Module.symvers needs to be shared between all KAL-client modules
      ln -sf "${moduleBuildDir}/../Module.symvers" "${moduleBuildDir}/Module.symvers"

      installFile Makefile         "${nikalDir}/src/client" "${moduleBuildDir}"
      installFile processmodule.sh "${nikalDir}/src/client" "${moduleBuildDir}"

      # Generate module-specific Makefile include
      nikalGetModuleGroupDir $moduleName
      cat <<-MAKE_MOD_INCLUDE > ${moduleBuildDir}/module.mak
			CLIENT_MODULE=$moduleUnversioned
			CLIENT_MOD_DIR=$moduleGroupDir
			CLIENT_NAME=$moduleName
			MAKE_MOD_INCLUDE
   fi

   test -d "${moduleBuildDir}"
   return $?
}

# Look up the path to the base directory for NI-KAL shared state files
# Inputs:
# Outputs:
#     nikalSharedStateDir - the directory where KAL can keep it's state files
# Returns: 0=success (cannot fail)
function nikalGetSharedStateDir() {
   nikalSharedStateDir="/var/lib/nikal"
   return 0
}

# Look up the path to the specified kernel module's build directory
# Inputs:
#     $1              - kernel module name
# Outputs:
#     moduleBuildDir  - the path to the kernel module build directory
# Returns: 0=success, nonzero=failure
function nikalGetModuleBuildDir() {
   local moduleName=$1

   # Test if kernelVersion is already set, otherwise, look it up
   if [ -z "$kernelVersion" ] && ! nikalGetRequestedKernelVersion; then
      return $?
   fi

   # Test if nikalSharedStateDir is already set, otherwise, look it up
   if [ -z "$nikalSharedStateDir" ] && ! nikalGetSharedStateDir; then
      return $?
   fi
   moduleBuildDir="${nikalSharedStateDir}/${kernelVersion}/${moduleName}"
   return 0
}

# Build the requested kernel module
# Inputs:
#     $1              - kernel module name
#     $2              - the path to the KAL client module -unversioned.o
#                       (unnecessary if kernel module name is "nikal")
#     kernelVersion   - the name of the configured kernel sources used for build
#     nikalDir        - the path where NI-KAL is installed
#     forceRebuild    - if set to zero, module will be forced to rebuild
# Outputs:
# Returns: 0=success, 255=nothing to do, 1=failure
# Requires a versioning-capable environment
function nikalBuildKernelModule() {
   if ! requirecommands make; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   local moduleName=$1
   local moduleUnversioned=$2
   local moduleBuildDir=
   local makeflags=

   if [[ -z "$nikalDir" ]]; then
      moduleBuildDir="client"
   else
      if ! nikalCreateModuleBuildDir $moduleName $moduleUnversioned; then
         error "Failed while attempting to set up $moduleName kernel module build tree"
         return 1
      fi
   fi

   # If the requested module is nikal, it may need to be configured before it can be built
   if [[ "${moduleName}" == "nikal" ]]; then
      # Makefile.in needs to be generated if it doesn't exist or isn't readable, or
      # if forced rebuilds have been requested
      if [[ ! -r "${moduleBuildDir}/Makefile.in" ]] || [[ "${forceRebuild}" = "0" ]]; then
         info "Configuring NI-KAL for kernel version ${kernelVersion}..."
         local logfile="/tmp/nikal${kernelVersion}Configure.log"
         if !(cd $moduleBuildDir && ./configure > "${logfile}"); then
            errorfile "${logfile}"
            rm -f  "${logfile}"
            error "failed to configure NI-KAL for kernel version ${kernelVersion}"
            return 1
         fi
         rm -f "${logfile}"
      fi
   fi

   # If forceRebuild is set, then tell Make to rebuild all targets,
   # regardless of freshness
   if [[ "${forceRebuild}" = "0" ]]; then
      makeflags="--always-make"
   else
      # If we're not forcing a rebuild, then check if the module is already up-to-date
      # and if so, short-circuit out
      (cd "$moduleBuildDir" && make --question install > /dev/null 2>&1)
      local makeAnswer=$?
      if [[ ${makeAnswer} = "0" ]]; then
         # Nothing to do
         info "Module ${moduleName} is up-to-date"
         return 255
      fi
   fi

   info "Building module ${moduleName}..."
   local logfile="/tmp/nikal${moduleName}Configure.log"
   if !(cd "$moduleBuildDir" && make ${makeflags} > "${logfile}" 2>&1); then
      errorfile "${logfile}"
      rm -f "${logfile}"
      error " failed to build $moduleName"
      return 1
   fi
   rm -f "${logfile}"

   local logfile="/tmp/nikal${moduleName}Make.log"
   if !(cd "$moduleBuildDir" && make ${makeflags} install >>"${logfile}" 2>&1); then
      errorfile "${logfile}"
      rm -f "${logfile}"
      error " failed to install $moduleName"
      return 1
   else
      info " ${moduleName} successfully updated."
   fi
   rm -f  "${logfile}"

   return 0
}

# Update the linux kernel module indexes
# Inputs:
# Outputs:
# Returns: 0=success, nonzero=failure
# Requires a versioning-capable environment
function nikalUpdateKernelDepmod() {
   if ! requirecommands depmod; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   # Test if kernelVersion is already set, otherwise, look it up
   if [ -z "$kernelVersion" ] && ! nikalGetRequestedKernelVersion; then
      return $?
   fi

   local logfile="/tmp/nikalKernelInstallerDepmodOut.txt"
   if ! depmod -a $kernelVersion > "${logfile}" 2>&1; then
      error "Running depmod failed"
      error "Output from the depmod command:"
      errorfile  "${logfile}"
      rm -f  "${logfile}"
      return 1
   fi
   rm -f  "${logfile}"

   return 0
}

# Create a system report
# Inputs:
# Outputs:
# Returns: 0=success, nonzero=failure
function nikalCreateSystemReport() {
   if ! requirecommands gzip niSystemReport; then
      error "Some required tools are missing or were not found."
      return 1
   fi

   niSystemReport | gzip -f > /tmp/niSystemReport.out.gz 2>&1
   error " Include the file /tmp/niSystemReport.out.gz when contacting"
   error " National Instruments for support."
   return 0
}

# Test whether the current running kernel is the same as the kernel
# NI-KAL is being asked to build against.
# Inputs:
# Outputs:
# Returns: 0=configured is the same, nonzero=configured is not the same
function configuredForCurrentKernel() {
   local configuredForCurrent=0
   local notConfiguredForCurrent=1

   # Test if kernelVersion is already set, otherwise, look it up
   if [ -z "$kernelVersion" ] && ! nikalGetRequestedKernelVersion; then
      return $?
   fi

   local runningKernel=$(uname -r)

   if [ "${kernelVersion}" = "${runningKernel}" ]; then
      return $configuredForCurrent
   else
      return $notConfiguredForCurrent
   fi
}

# Test whether the specified kernel module is currently loaded
# Inputs:
#     $1              - kernel module name
# Outputs:
# Returns: 0=module is loaded, nonzero=module is not loaded or error
function testKernelDriverLoaded() {
   if ! requirecommands lsmod grep; then
      error "Some required tools are missing or were not found."
      return 1
   fi
   lsmod | grep "^$1" >/dev/null 2>&1
   return $?
}

# Unloads the specified kernel module, and all modules depending on it
# Inputs:
#     $1              - kernel module name
# Outputs:
#     nikalUnloadedModules - bash array of modules that were unloaded
# Returns: 0=module is unloaded, nonzero=module could not be unloaded or error
function recursiveUnloadKernelDriver() {
   nikalUnloadedModules=()
   # default returnStatus is success
   local returnStatus=0

   if testKernelDriverLoaded $1; then
      recursiveUnloadKernelDriverHelper $1
      returnStatus=$?
   fi
   return $returnStatus
}

# Unloads the specified kernel module, and all modules depending on it
# Inputs:
#     $1              - kernel module name
# Outputs:
# Returns: 0=module is unloaded, nonzero=module could not be unloaded or error
function recursiveUnloadKernelDriverHelper() {
   if ! requirecommands lsmod rmmod grep wc awk sed; then
      error "Some required tools are missing or were not found."
      return 1
   fi
   # if this module is depended upon by other modules, then stop those
   # first
   if [ $(lsmod | grep "^$1" | wc -w) = 4 ]; then
      for i in $(lsmod | grep "^$1" | awk '{ print $4 }' | sed 's/,/ /g'); do
         recursiveUnloadKernelDriverHelper $i
         if [ $? -ne 0 ]; then
            return 1
         fi
      done
   fi
   rmmod $1 >/dev/null 2>&1
   local rmmodStatus=$?
   if [ "$rmmodStatus" != "0" ]; then
      warning "Could not unload module $1 from memory"
   else
      nikalUnloadedModules+=($1)
   fi

   return $rmmodStatus
}

# Loads kernel modules that had been unloaded by recursiveUnloadKernelDriver
# Inputs:
#     $1              - bash array containing modules to be loaded
# Outputs:
# Returns: 0=module is unloaded, nonzero=module(s) could not be loaded or error
function reloadKernelDrivers() {
   local ret=0
   for mod in "${nikalUnloadedModules[@]}"; do
      if ! modprobe "$mod"; then
         ret=$?
         error "An error occurred while loading $mod."
      fi
   done
   return $ret
}

