//
// Copyright 2011 Ettus Research LLC
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
#define use_short
#include <uhd/utils/thread_priority.hpp>
#include <uhd/utils/safe_main.hpp>
#include <uhd/usrp/multi_usrp.hpp>
#include <boost/program_options.hpp>
#include <boost/format.hpp>
#include <boost/thread.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>
#include <iostream>
#include <fstream>
#include <complex>
#include <cmath>

namespace po = boost::program_options;




typedef boost::function<uhd::sensor_value_t (const std::string&)> get_sensor_fn_t;
bool check_locked_sensor(std::vector<std::string> sensor_names, const char* sensor_name, get_sensor_fn_t get_sensor_fn, double setup_time);


int UHD_SAFE_MAIN(int argc, char *argv[]){
    uhd::set_thread_priority_safe();

    //variables to be set by po
    std::string args, sync, subdev, channel_list,file,file0,file1;
    double seconds_in_future, freq;
    size_t total_num_samps;
    double rate,gain,tx_gain;

    //setup the program options
    po::options_description desc("Allowed options");
    desc.add_options()
        ("help", "help message")
        ("args", po::value<std::string>(&args)->default_value(""), "single uhd device address args")
        ("file", po::value<std::string>(&file)->default_value("measure"), "name of the file to write binary samples to")
        ("secs", po::value<double>(&seconds_in_future)->default_value(1.0), "number of seconds in the future to receive")
        ("nsamps", po::value<size_t>(&total_num_samps)->default_value(10000), "total number of samples to receive")
        ("rate", po::value<double>(&rate)->default_value(8e6), "rate of incoming samples")
        ("sync", po::value<std::string>(&sync)->default_value("b210"), "synchronization method: now, pps, mimo")
        ("freq", po::value<double>(&freq)->default_value(800e6), "RF center frequency in Hz")
        ("gain", po::value<double>(&gain), "gain for the RF chain")
        ("tx-gain", po::value<double>(&tx_gain), "gain for the TX in calibration Mode")
        ("subdev", po::value<std::string>(&subdev)->default_value("A:A A:B"),"subdev spec (homogeneous across motherboards)")
        ("v", "specify to enable inner-loop verbose")
        ("calc","specify to enable transmitting and recording of calibration data")
        //unused("channels", po::value<std::string>(&channel_list)->default_value("0"), "which channel(s) to use (specify \"0\", \"1\", \"0,1\", etc)")
    ;
    po::variables_map vm;
    po::store(po::parse_command_line(argc, argv, desc), vm);
    po::notify(vm);

    //print the help message
    if (vm.count("help")){
        std::cout << boost::format("UHD RX Multi Samples %s") % desc << std::endl;
        std::cout <<
        "    This is a demonstration of how to receive aligned data from multiple channels.\n"
        "    This example can receive from multiple DSPs, multiple motherboards, or both.\n"
        "    The MIMO cable or PPS can be used to synchronize the configuration. See --sync\n"
        "\n"
        "    Specify --subdev to select multiple channels per motherboard.\n"
        "      Ex: --subdev=\"0:A 0:B\" to get 2 channels on a Basic RX.\n"
        "\n"
        "    Specify --args to select multiple motherboards in a configuration.\n"
        "      Ex: --args=\"addr0=192.168.10.2, addr1=192.168.10.3\"\n"
        << std::endl;
        return ~0;
    }

    bool verbose = vm.count("v") == 1;
    bool calibration = vm.count("calc") == 1;
    


    /**************** Main Init ************************************************/
    //create a usrp device
    std::cout << std::endl;
    std::cout << boost::format("Creating the usrp device with: %s...") % args << std::endl;
    uhd::usrp::multi_usrp::sptr usrp = uhd::usrp::multi_usrp::make(args);
    std::cout.flush();
    std::cout << std::endl;

    //Lock mboard clocks
    std::string ref;
    double setup_time=1.0;
    ref = "internal";
    //ref = "external";
    double clock = 2*rate*floor(30e6/(2*rate));
    usrp->set_master_clock_rate(clock); //less then 30.72e6 //Möglichkeit 1: Variable Clock-Rate
    usrp->set_clock_source(ref);

    boost::this_thread::sleep(boost::posix_time::seconds(setup_time)); //allow for some setup time

    //check Ref and LO Lock detect
    if (not vm.count("skip-lo")){
        check_locked_sensor(usrp->get_rx_sensor_names(0), "lo_locked", boost::bind(&uhd::usrp::multi_usrp::get_rx_sensor, usrp, _1, 0), setup_time);
        if (ref == "mimo")
            check_locked_sensor(usrp->get_mboard_sensor_names(0), "mimo_locked", boost::bind(&uhd::usrp::multi_usrp::get_mboard_sensor, usrp, _1, 0), setup_time);
        if (ref == "external"){
            /* eigener Test
            std::vector< std::string > 	sens = usrp->get_mboard_sensor_names();
            std::cout << "Mboard Sensor " << sens[0] << " mit Status " << usrp->get_mboard_sensor(sens[0]).to_pp_string() << std::endl;
            std::cout.flush();*/
            
            check_locked_sensor(usrp->get_mboard_sensor_names(0), "ref_locked", boost::bind(&uhd::usrp::multi_usrp::get_mboard_sensor, usrp, _1, 0), setup_time);
        }
    }

    /**************** RX Init *************************************************/
    //always select the subdevice first, the channel mapping affects the other settings
    if (vm.count("subdev")){
        uhd::usrp::subdev_spec_t subdevClass = subdev;
        std::cout << subdevClass.to_pp_string() << std::endl;
        usrp->set_rx_subdev_spec(subdevClass); //sets across all mboards
    }

    std::cout << boost::format("Using Device: %s") % usrp->get_pp_string() << std::endl;

    size_t num_chan = usrp->get_rx_num_channels(); //TODO: Get evtl. von Anzahl an Subdev

    //linearly map channels (index0 = channel0, index1 = channel1, ...)
    std::vector<size_t>channel_nums;
    for (size_t i = 0; i < num_chan; i++)
        channel_nums.push_back(i);

    //set the rx sample rate (sets across all channels)
    std::cout << boost::format("Setting RX Rate: %f Msps... ") % (rate/1e6) << std::endl;
    usrp->set_rx_rate(rate);
    std::cout << boost::format("Actual RX Rate: %f Msps...") % (usrp->get_rx_rate()/1e6) << std::endl << std::endl;

    //set center Frequency
    std::cout << boost::format("Setting RX Freq: %f MHz...") % (freq/1e6) << std::endl;
    uhd::tune_request_t tune_request(freq);
    for(size_t i=0;i<num_chan;i++)
        usrp->set_rx_freq(tune_request,channel_nums[i]);
    std::cout << boost::format("Actual RX Freq: %f MHz...") % (usrp->get_rx_freq(1)/1e6) << std::endl << std::endl;

    //set the rf gain
    if (vm.count("gain")){
        for(size_t i=0;i<num_chan;i++){
            std::cout << boost::format("Setting RX Gain for Channel %d: %f dB...") % channel_nums[i] % gain << std::endl;
            usrp->set_rx_gain(gain,channel_nums[i]);
            std::cout << boost::format("Actual RX Gain: %f dB...") % usrp->get_rx_gain() << std::endl << std::endl;
        }
    }

    //open files to write
    std::ofstream outfile[num_chan];
    for (size_t i=0; i<num_chan; i++)
        outfile[i].open((boost::format("%s%u") % file % i).str().c_str(), std::ofstream::binary);



    /**************** Recieve Start ******************************************/
    //create a receive streamer
    uhd::stream_args_t stream_args("sc16"); //complex short
    stream_args.channels = channel_nums;
    uhd::rx_streamer::sptr rx_stream = usrp->get_rx_stream(stream_args);

    //setup streaming
    std::cout << std::endl;
    std::cout << boost::format(
        "Begin receiving of %u samples. Start in %f seconds in the future..."
    ) % total_num_samps % seconds_in_future << std::endl;
    uhd::stream_cmd_t stream_cmd(uhd::stream_cmd_t::STREAM_MODE_NUM_SAMPS_AND_DONE);
    stream_cmd.num_samps = total_num_samps;
    stream_cmd.stream_now = false;
    stream_cmd.time_spec = uhd::time_spec_t(seconds_in_future) + usrp->get_time_now();
    rx_stream->issue_stream_cmd(stream_cmd); //tells all channels to stream

    //meta-data will be filled in by recv()
    uhd::rx_metadata_t md;

    //allocate buffers to receive with samples (one buffer per channel)
    const size_t samps_per_buff = rx_stream->get_max_num_samps();
    std::vector<std::vector<std::complex<short> > > buffs(
        num_chan, std::vector<std::complex<short> >(samps_per_buff)
    );

    //create a vector of pointers to point to each of the channel buffers
    std::vector<std::complex<short> *> buff_ptrs;
    for (size_t i = 0; i < buffs.size(); i++) buff_ptrs.push_back(&buffs[i].front());

    //the first call to recv() will block this many seconds before receiving
    double timeout = seconds_in_future + 1; //timeout (delay before receive + padding)

    size_t num_acc_samps = 0; //number of accumulated samples
    bool had_an_overflow = false;
    uhd::time_spec_t last_time;
    unsigned long long num_overflows = 0;
    unsigned long long num_rx_samps = 0;
    unsigned long long num_dropped_samps = 0;

    unsigned long long last_overflow_num_samps = 0;
    
    while(num_acc_samps < total_num_samps){

        //receive a single packet
        num_rx_samps = rx_stream->recv(buff_ptrs, samps_per_buff, md, timeout);

        

        //use a small timeout for subsequent packets
        timeout = 0.1;

        //handle the error code
        switch(md.error_code){
        case uhd::rx_metadata_t::ERROR_CODE_NONE:
            if (had_an_overflow){
                had_an_overflow = false;
                num_dropped_samps += (md.time_spec - last_time).to_ticks(rate);
            }
            break;

        // ERROR_CODE_OVERFLOW can indicate overflow or sequence error
        case uhd::rx_metadata_t::ERROR_CODE_OVERFLOW:
            last_time = md.time_spec;
            last_overflow_num_samps = num_rx_samps;
            had_an_overflow = true;
            // check out_of_sequence flag to see if it was a sequence error or overflow
            if (!md.out_of_sequence)
                num_overflows++;
            break;

        default:
            std::cerr << "Receiver error: " << md.strerror() << std::endl;
            std::cerr << "Unexpected error on recv, continuing..." << std::endl;
            break;
        }
        /*if (md.error_code == uhd::rx_metadata_t::ERROR_CODE_TIMEOUT) break;
        if (md.error_code != uhd::rx_metadata_t::ERROR_CODE_NONE){
            throw std::runtime_error(str(boost::format(
                "Receiver error %s"
            ) % md.strerror()));
        }*/

        if(verbose) std::cout << boost::format(
            "Received packet: %u samples, %u full secs, %f frac secs"
        ) % num_rx_samps % md.time_spec.get_full_secs() % md.time_spec.get_frac_secs() << std::endl;

        // write the data to the output files
        for (size_t i=0; i<num_chan; i++)
            if (outfile[i].is_open())
                outfile[i].write((const char*)buff_ptrs[i], num_rx_samps*sizeof(buffs[0].front()));

    

        num_acc_samps += num_rx_samps;
    }

    if (num_acc_samps < total_num_samps) std::cerr << "Receive timeout before all samples received..." << std::endl;



    /**************************** Clean up **********************************************************/
    for(size_t i=0; i<num_chan;i++)
        if(outfile[i].is_open())
            outfile[i].close();

    //print summary
    std::cout << std::endl << boost::format(
        "Summary:\n"
        "  Num received samples:    %u\n"
        "  Num dropped samples:     %u\n"
        "  Num overflows detected:  %u\n"
        "  - Last overflow sample num: %u\n"
    ) % num_acc_samps % num_dropped_samps % num_overflows % last_overflow_num_samps  << std::endl;
    
    //finished
    std::cout << std::endl << "Done! Mine" << std::endl << std::endl;

    return EXIT_SUCCESS;
}

bool check_locked_sensor(std::vector<std::string> sensor_names, const char* sensor_name, get_sensor_fn_t get_sensor_fn, double setup_time){
    if (std::find(sensor_names.begin(), sensor_names.end(), sensor_name) == sensor_names.end())
        return false;

    boost::system_time start = boost::get_system_time();
    boost::system_time first_lock_time;

    std::cout << boost::format("Waiting for \"%s\": ") % sensor_name;
    std::cout.flush();

    while (true){
        if ((not first_lock_time.is_not_a_date_time()) and
            (boost::get_system_time() > (first_lock_time + boost::posix_time::seconds(setup_time))))
        {
            std::cout << " locked." << std::endl;
            break;
        }

        if (get_sensor_fn(sensor_name).to_bool()){
            if (first_lock_time.is_not_a_date_time())
                first_lock_time = boost::get_system_time();
            std::cout << "+";
            std::cout.flush();
        }
        else{
            first_lock_time = boost::system_time(); //reset to 'not a date time'

            if (boost::get_system_time() > (start + boost::posix_time::seconds(setup_time))){
                std::cout << std::endl;
                throw std::runtime_error(str(boost::format("timed out waiting for consecutive locks on sensor \"%s\"") % sensor_name));
            }

            std::cout << "_";
            std::cout.flush();
        }

        boost::this_thread::sleep(boost::posix_time::milliseconds(100));
    }

    std::cout << std::endl;

    return true;
}
